<?php
/**
 * Compatibility with YITH WooCommerce Multivendor
 *
 * @package YITH WooCommerce Dynamic Pricing and Discounts Premium
 * @since   1.0.0
 * @version 1.6.0
 * @author  YITH
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * YWDPD_Multivendor class to add compatibility with YITH WooCommerce Multivendor
 *
 * @class   YWDPD_Multivendor
 * @package YITH WooCommerce Dynamic Pricing and Discounts
 * @since   1.0.0
 * @author  YITH
 */
if ( ! class_exists( 'YITH_YWDPD_Multi_Vendor_Integration' ) ) {

	/**
	 * Class YITH_YWDPD_Multi_Vendor_Integration
	 */
	class YITH_YWDPD_Multi_Vendor_Integration {

		/**
		 * Single instance of the class
		 *
		 * @var YITH_YWDPD_Multi_Vendor_Integration
		 */
		protected static $instance;


		/**
		 * Returns single instance of the class
		 *
		 * @return YITH_YWDPD_Multi_Vendor_Integration
		 * @since 1.0.0
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Constructor
		 *
		 * Initialize class and registers actions and filters to be used
		 *
		 * @since  1.0.0
		 * @author Emanuela Castorina
		 */
		public function __construct() {

			add_filter( 'ywdpd_custom_taxonomy', array( $this, 'add_vendor_taxonomy_in_counters' ), 10, 1 );

			// PRICE OPTIONS.
			add_filter( 'ywdpd_vendor_options_rule_for', array( $this, 'add_pricing_rule_option' ) );
			add_filter( 'ywdpd_vendor_options_exclude_rule_for', array( $this, 'add_pricing_rule_option' ) );
			add_filter( 'ywdpd_vendor_apply_adjustment_rule_for', array( $this, 'add_pricing_rule_option' ) );
			add_filter( 'ywdpd_vendor_apply_exclude_adjustment_rule_for', array( $this, 'add_pricing_rule_option' ) );
			add_filter( 'yit_ywdpd_pricing_rules_options', array( $this, 'add_pricing_option_rule_for' ) );

			// Valid Price rules.
			add_filter( 'ywdpd_is_valid_to_apply', array( $this, 'valid_rule_for_vendor' ), 20, 4 );
			add_filter( 'ywdpd_is_product_excluded_from_apply', array( $this, 'product_excluded_for_vendor' ), 20, 4 );
			add_filter( 'ywdpd_is_valid_to_adjust', array( $this, 'valid_rule_adjustment_to_vendor' ), 20, 4 );

			// CART RULES.
			add_filter( 'ywdpd_cart_rules_product_include_fields', array( $this, 'add_cart_rule_option' ) );
			add_filter( 'ywdpd_cart_rules_product_exclude_fields', array( $this, 'add_cart_rule_option' ) );
			add_filter( 'ywdpd_valid_product_cart_condition', array( $this, 'yith_vendor_valid_require_vendor_condition' ), 20, 4 );
			add_filter( 'ywpd_is_excluded_product_cart_condition', array( $this, 'yith_vendor_valid_exclude_vendor_condition' ), 20, 4 );

			// SPECIAL OFFER IN POPUP.

			add_filter( 'ywdpd_involved_items', array( $this, 'add_vendor_args_in_item' ), 20, 3 );
			add_filter( 'ywdpd_involved_adjustment_items', array( $this, 'add_vendor_adjustment_args_in_item' ), 20, 3 );
			add_filter( 'ywdpd_get_product_ids_involved_to_exclude', array( $this, 'return_product_ids_to_exclude' ), 20, 3 );
			add_filter( 'ywdpd_get_product_ids_involved_adjustment_to_exclude', array( $this, 'return_product_ids_to_exclude' ), 20, 3 );
			add_filter( 'ywdpd_get_comulative_quantity', array( $this, 'check_comulative_quantity' ), 20, 3 );
		}

		/**
		 * Add the vendor taxonomy in counter
		 *
		 * @param array $custom_taxonomy The custom taxonomy.
		 *
		 * @return array
		 * @author YITH
		 * @since 3.0.0
		 */
		public function add_vendor_taxonomy_in_counters( $custom_taxonomy ) {

			$custom_taxonomy[] = YITH_Vendors()->get_taxonomy_name();

			return $custom_taxonomy;
		}

		/**
		 * Check if the rule is valid
		 *
		 * @param bool             $is_valid If rule is valid.
		 * @param string           $type The rule product type.
		 * @param WC_Product       $product The product.
		 * @param YWDPD_Price_Rule $rule The rule.
		 *
		 * @return bool
		 * @since 3.0.0
		 * @author YITH
		 */
		public function valid_rule_for_vendor( $is_valid, $type, $product, $rule ) {

			if ( 'vendor_list' === $type ) {

				$vendor_list_ids = $rule->get_meta( 'apply_to_vendors_list' );
				$is_valid        = $rule->is_product_in_list( $product, $vendor_list_ids, YITH_Vendors()->get_taxonomy_name() );
			}

			return $is_valid;
		}

		/**
		 * Check if this product is excluded from a rule
		 *
		 * @param bool             $is_excluded If is excluded.
		 * @param string           $type The rule product type.
		 * @param WC_Product       $product The product.
		 * @param YWDPD_Price_Rule $rule The rule.
		 *
		 * @return bool
		 * @since 3.0.0
		 * @author YITH
		 */
		public function product_excluded_for_vendor( $is_excluded, $type, $product, $rule ) {

			if ( 'vendor_list_excluded' === $type && $rule->is_exclude_from_apply_enabled() ) {
				$vendor_list_ids = $rule->get_meta( 'apply_to_vendors_list_excluded' );
				$is_excluded     = $rule->is_product_in_list( $product, $vendor_list_ids, YITH_Vendors()->get_taxonomy_name() );
			}

			return $is_excluded;
		}

		/**
		 * Check if the rule is valid to adjustment to
		 *
		 * @param bool             $is_valid Is valid.
		 * @param string           $type The check type.
		 * @param WC_Product       $product The product.
		 * @param YWDPD_Price_Rule $rule The price rule.
		 *
		 * @return bool
		 * @since 3.0.0
		 * @author YITH
		 */
		public function valid_rule_adjustment_to_vendor( $is_valid, $type, $product, $rule ) {
			$is_excluded      = false;
			$is_active_adjust = $rule->is_enabled_apply_adjustment_to();
			if ( $is_active_adjust ) {

				if ( $rule->is_exclude_adjustment_to_enabled() && 'vendor_list_excluded' === $rule->get_exclude_apply_adjustment_rule_for() ) {
					$vendor_list_ids = $rule->get_meta( 'apply_adjustment_vendor_list_excluded' );
					$is_excluded     = $rule->is_product_in_list( $product, $vendor_list_ids, YITH_Vendors()->get_taxonomy_name() );
				}
				if ( ! $is_excluded && 'vendor_list' === $rule->get_rule_apply_adjustment_discount_for() ) {
					$vendor_list_ids = $rule->get_meta( 'apply_adjustment_vendor_list' );
					$is_valid        = $rule->is_product_in_list( $product, $vendor_list_ids, YITH_Vendors()->get_taxonomy_name() );
				}
			}

			return $is_valid;
		}

		/**
		 * Add the vendor fields in metabox price rule
		 *
		 * @param array $rules The rules.
		 *
		 * @return array
		 */
		public function add_pricing_option_rule_for( $rules ) {
			$new_rule = array();
			foreach ( $rules as $key => $rule ) {
				$new_rule[ $key ] = $rule;

				if ( 'rule_for' === $key || 'rule_apply_adjustment_discount_for' === $key ) {
					$new_rule[ $key ]['vendor_list'] = __( 'Specific vendors', 'ywdpd' );

				}

				if ( 'exclude_rule_for' === $key || 'exclude_apply_adjustment_rule_for' === $key ) {
					$new_rule[ $key ]['vendor_list_excluded'] = __( 'Specific vendors', 'ywdpd' );
				}
			}

			return $new_rule;
		}

		/**
		 * Add pricing rules options in settings panels
		 *
		 * @return array
		 */
		public function add_pricing_rule_option() {

			$current_filter = current_filter();
			if ( 'ywdpd_vendor_options_rule_for' === $current_filter ) {
				$option = array(
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => YITH_Vendors()->get_taxonomy_name(),
						'placeholder' => __( 'Search for a vendor', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_rule_for',
									'value' => 'vendor_list',
								),
								array(
									'id'    => '_discount_mode',
									'value' => 'bulk,special_offer,gift_products,bogo',
								),
							)
						),
					),
					'label'    => __( 'Apply rule to:', 'ywdpd' ),
					'desc'     => __( 'Search the vendor(s) to include in the rule', 'ywdpd' ),
					'multiple' => true,
				);
			} elseif ( 'ywdpd_vendor_options_exclude_rule_for' === $current_filter ) {
				$option = array(
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => YITH_Vendors()->get_taxonomy_name(),
						'placeholder' => __( 'Search for a vendor', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_active_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_exclude_rule_for',
									'value' => 'vendor_list_excluded',
								),
							)
						),
					),
					'label'    => __( 'Choose which vendor(s) to exclude', 'ywdpd' ),
					'desc'     => __( 'Search the vendor(s) to exclude from the rule', 'ywdpd' ),
					'multiple' => true,
				);
			} elseif ( 'ywdpd_vendor_apply_adjustment_rule_for' === $current_filter ) {
				$option = array(
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => YITH_Vendors()->get_taxonomy_name(),
						'placeholder' => __( 'Search for a vendor', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_rule_apply_adjustment_discount_for',
									'value' => 'vendor_list',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
								array(
									'id'    => '_discount_mode',
									'value' => 'bulk,special_offer',
								),
							)
						),
					),
					'label'    => __( 'Choose which vendor(s) to include', 'ywdpd' ),
					'desc'     => __( 'Search the vendor(s) to include in this discount', 'ywdpd' ),
					'multiple' => true,
				);
			} else {
				$option = array(
					'type'        => 'ajax-terms',
					'data'        => array(
						'taxonomy'    => YITH_Vendors()->get_taxonomy_name(),
						'placeholder' => __( 'Search for a vendor', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_exclude_apply_adjustment_rule_for',
									'value' => 'vendor_list_excluded',
								),
								array(
									'id'    => '_active_apply_adjustment_to_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_apply_adjustment_discount_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
								array(
									'id'    => '_discount_mode',
									'value' => 'bulk,special_offer',
								),
							)
						),
					),
					'label'       => __( 'Choose which vendor(s) to exclude', 'ywdpd' ),
					'placeholder' => __( 'Search for a vendor', 'ywdpd' ),
					'desc'        => __( 'Search the vendor(s) to exclude from this discount', 'ywdpd' ),
					'multiple'    => true,
				);
			}

			return $option;
		}


		/**
		 * Add cart rules options in settings panels
		 *
		 * @param array $rules The cart rule options.
		 *
		 * @return array
		 */
		public function add_cart_rule_option( $rules ) {

			if ( 'ywdpd_cart_rules_product_include_fields' === current_filter() ) {

				$options = array(
					array(
						'id'        => 'enable_require_product_vendors',
						'name'      => __( 'Require specific vendors in cart', 'ywdpd' ),
						'desc'      => __( 'Enable to require products of specific vendors in cart to apply the discount', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product require_product',
						'class'     => 'ywdpd_enable_require_product_vendors',
					),
					array(
						'id'        => 'require_product_vendors_list',
						'name'      => __( 'Include a list of vendors', 'ywdpd' ),
						'desc'      => __( 'Choose which vendors are required in cart to apply the discount', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'    => YITH_Vendors()->get_taxonomy_name(),
							'placeholder' => __( 'Search for vendor', 'ywdpd' ),
						),
						'class_row' => 'product require_product enable_require_product_vendors_list',
						'multiple'  => true,
					),
				);
				$rules   = array_merge( $rules, $options );
			} elseif ( 'ywdpd_cart_rules_product_exclude_fields' === current_filter() ) {
				$options = array(
					array(
						'id'        => 'enable_exclude_product_vendors',
						'name'      => __( 'Exclude specific vendors from discount validation', 'ywdpd' ),
						'desc'      => __( 'Enable if you want to exclude specific vendors to this cart discount validation', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product exclude_product',
						'class'     => 'ywdpd_enable_exclude_product_vendors',
					),
					array(
						'id'        => 'exclude_product_vendors_list',
						'name'      => __( 'Exclude a list of vendors', 'ywdpd' ),
						'desc'      => __( 'Choose which product vendors to exclude from this cart validation', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'    => YITH_Vendors()->get_taxonomy_name(),
							'placeholder' => __( 'Search for vendor', 'ywdpd' ),
						),
						'class_row' => 'product exclude_product enable_exclude_product_vendors_list',
						'multiple'  => true,
					),
				);
				$rules   = array_merge( $rules, $options );
			}

			return $rules;
		}


		/**
		 * Valid the cart rule for vendors
		 *
		 * @param bool            $is_valid If rule is valid.
		 * @param array           $condition The specific condition.
		 * @param array           $conditions All conditions.
		 * @param YWDPD_Cart_Rule $rule The cart rule.
		 *
		 * @return bool
		 */
		public function yith_vendor_valid_require_vendor_condition( $is_valid, $condition, $conditions, $rule ) {

			$type_check = $rule->get_condition_product_type( $condition );

			if ( ! $is_valid && 'require_product' === $type_check ) {
				$require_product_vendor = yith_plugin_fw_is_true( $condition['enable_require_product_vendors'] );
				$vendor_to_check        = isset( $condition['require_product_vendors_list'] ) ? $condition['require_product_vendors_list'] : array();

				if ( $require_product_vendor && is_array( $vendor_to_check ) && count( $vendor_to_check ) > 0 ) {

					$is_valid = $rule->check_taxonomy_in_cart( $vendor_to_check, YITH_Vendors()->get_taxonomy_name(), 'at_least' );
				}
			}

			return $is_valid;
		}

		/**
		 * Check if the vendor is excluded
		 *
		 * @param bool            $is_excluded If the rule is excluded.
		 * @param array           $condition The condition.
		 * @param WC_Product      $product The product.
		 * @param YWDPD_Cart_Rule $rule The rule.
		 *
		 * @return bool
		 */
		public function yith_vendor_valid_exclude_vendor_condition( $is_excluded, $condition, $product, $rule ) {

			$exclude_product_vendor = yith_plugin_fw_is_true( $condition['enable_exclude_product_vendors'] );
			$vendor_id_to_exclude   = isset( $condition['exclude_product_vendors_list'] ) ? $condition['exclude_product_vendors_list'] : array();

			if ( ! $is_excluded ) {
				if ( $exclude_product_vendor && count( $vendor_id_to_exclude ) > 0 ) {
					$product_terms = YWDPD_Taxonomy::get_product_term_ids( $product, YITH_Vendors()->get_taxonomy_name() );
					$is_excluded   = count( array_intersect( $product_terms, $vendor_id_to_exclude ) ) > 0;
				}
			}

			return $is_excluded;
		}

		/**
		 * Add the vendor taxonomy in popup.
		 *
		 * @param array            $items The args.
		 * @param YWDPD_Price_Rule $price_rule The price rule.
		 * @param string           $rule_for The rule for.
		 *
		 * @return array
		 */
		public function add_vendor_args_in_item( $items, $price_rule, $rule_for ) {

			if ( 'vendor_list' === $rule_for ) {
				$items = array(
					'item_ids' => $price_rule->get_meta( 'apply_to_vendors_list' ),
					'type'     => YITH_Vendor::$taxonomy,
				);
			}

			return $items;
		}

		/**
		 * Add the vendor taxonomy in popup.
		 *
		 * @param array            $items The args.
		 * @param YWDPD_Price_Rule $price_rule The price rule.
		 * @param string           $rule_for The rule for.
		 *
		 * @return array
		 */
		public function add_vendor_adjustment_args_in_item( $items, $price_rule, $rule_for ) {

			if ( 'vendor_list' === $rule_for ) {
				$items = array(
					'item_ids' => $price_rule->get_meta( 'apply_adjustment_vendor_list' ),
					'type'     => YITH_Vendor::$taxonomy,
				);
			}

			return $items;
		}

		/**
		 * Return the product id to exclude
		 *
		 * @param array            $product_ids The product ids.
		 * @param YWDPD_Price_Rule $rule The cart rule.
		 * @param string           $exclude_for The exclusion type.
		 *
		 * @return array
		 * @author YITH
		 * @since 3.0.0
		 */
		public function return_product_ids_to_exclude( $product_ids, $rule, $exclude_for ) {

			if ( 'vendor_list_excluded' === $exclude_for ) {
				if ( 'ywdpd_get_product_ids_involved_to_exclude' === current_filter() ) {
					$vendor_ids = $rule->get_meta( 'apply_adjustment_vendor_list_excluded' );
				} else {
					$vendor_ids = $rule->get_meta( 'apply_to_vendors_list_excluded' );
				}
				if ( is_array( $vendor_ids ) && count( $vendor_ids ) > 0 ) {
					$tax_query = WC()->query->get_tax_query(
						array(
							array(
								'taxonomy' => YITH_Vendor::$taxonomy,
								'terms'    => array_values( $vendor_ids ),
								'operator' => 'IN',
							),
						)
					);

					$product_ids = get_posts(
						array(
							'numberposts' => - 1,
							'post_type'   => array( 'product' ),
							'post_status' => 'publish',
							'tax_query'   => $tax_query, // phpcs:ignore WordPress.DB.SlowDBQuery
							'fields'      => 'ids',
						)
					);
				}
			}

			return $product_ids;
		}

		/**
		 * Add or remove the quantity for vendor products
		 *
		 * @param int              $qty The comulative quantity.
		 * @param string           $rule_type The rule type.
		 * @param YWDPD_Price_Rule $rule The rule.
		 *
		 * @return int
		 * @author YITH
		 * @since 3.0.0
		 */
		public function check_comulative_quantity( $qty, $rule_type, $rule ) {
			$qt_ex = 0;
			if ( 'vendor_list' === $rule_type ) {
				$list_ids = $rule->get_meta( 'apply_to_vendors_list' );
				$qty      = YWDPD_Counter::get_comulative_taxonomy_count( $list_ids, YITH_Vendors()->get_taxonomy_name() );
			}
			$is_exclude_activated = $rule->is_exclude_from_apply_enabled();

			if ( $is_exclude_activated ) {
				$ex_type = $rule->get_exclude_rule_for();

				if ( 'vendor_list_excluded' === $ex_type ) {
					$list_ids = $rule->get_meta( 'apply_to_vendors_list_excluded' );
					$qt_ex    = YWDPD_Counter::get_comulative_taxonomy_count( $list_ids, YITH_Vendors()->get_taxonomy_name() );

				}
			}

			return $qty - $qt_ex;
		}

	}
}
