<?php
/**
 * Compatibility with YITH WooCommerce Brands Add-on Premium
 *
 * @package YITH WooCommerce Dynamic Pricing and Discounts Premium
 * @since   1.0.0
 * @version 1.6.0
 * @author  YITH
 */

if ( ! defined( 'ABSPATH' ) || ! defined( 'YITH_YWDPD_VERSION' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * YWDPD_Brands class to add compatibility with YITH WooCommerce Brands Add-on Premium
 *
 * @class   YWDPD_Brands
 * @package YITH WooCommerce Dynamic Pricing and Discounts
 * @since   1.1.7
 * @author  YITH
 */
if ( ! class_exists( 'YITH_YWDPD_Brands_Addon_Integration' ) ) {

	/**
	 * Class YITH_YWDPD_Brands_Addon_Integration
	 */
	class YITH_YWDPD_Brands_Addon_Integration {

		/**
		 * Single instance of the class
		 *
		 * @var YITH_YWDPD_Brands_Addon_Integration
		 */
		protected static $instance;

		/**
		 * Returns single instance of the class
		 *
		 * @return YITH_YWDPD_Brands_Addon_Integration
		 * @since 1.0.0
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Constructor
		 *
		 * Initialize class and registers actions and filters to be used
		 *
		 * @since  1.1.7
		 * @author Emanuela Castorina
		 */
		public function __construct() {

			add_filter( 'ywdpd_custom_taxonomy', array( $this, 'add_brands_taxonomy_in_counters' ), 10, 1 );

			// PRICE RULES.
			add_filter( 'yit_ywdpd_pricing_rules_options', array( $this, 'add_pricing_rule_option_rule_for' ) );
			add_filter( 'ywdpd_brand_options_rule_for', array( $this, 'add_pricing_rule_option' ) );
			add_filter( 'ywdpd_brand_options_exclude_rule_for', array( $this, 'add_pricing_rule_option' ) );
			add_filter( 'ywdpd_brand_apply_adjustment_rule_for', array( $this, 'add_pricing_rule_option' ) );
			add_filter( 'ywdpd_brand_apply_exclude_adjustment_rule_for', array( $this, 'add_pricing_rule_option' ) );

			// Valid price rule.

			add_filter( 'ywdpd_is_valid_to_apply', array( $this, 'valid_rule_for_brands' ), 21, 4 );
			add_filter( 'ywdpd_is_product_excluded_from_apply', array( $this, 'product_excluded_for_brands' ), 20, 4 );
			add_filter( 'ywdpd_get_comulative_quantity', array( $this, 'brands_comulative_quantity' ), 20, 3 );
			add_filter( 'ywdpd_is_valid_to_adjust', array( $this, 'valid_rule_adjustment_to_brands' ), 20, 4 );

			// CART RULES.
			add_filter( 'ywdpd_cart_rules_product_include_fields', array( $this, 'add_cart_rule_option' ) );
			add_filter( 'ywdpd_cart_rules_product_exclude_fields', array( $this, 'add_cart_rule_option' ) );
			add_filter( 'ywdpd_cart_rules_product_disable_fields', array( $this, 'add_cart_rule_option' ) );

			add_filter( 'ywdpd_valid_product_cart_condition', array( $this, 'valid_product_cart_condition' ), 15, 4 );
			add_filter( 'ywpd_is_excluded_product_cart_condition', array( $this, 'valid_product_exclude_cart_condition' ), 15, 3 );

			// SPECIAL OFFER IN POPUP.
			add_filter( 'ywdpd_involved_items', array( $this, 'add_brands_args_in_item' ), 20, 3 );
			add_filter( 'ywdpd_involved_adjustment_items', array( $this, 'add_brands_adjustment_args_in_item' ), 20, 3 );
			add_filter( 'ywdpd_get_product_ids_involved_to_exclude', array( $this, 'return_product_ids_to_exclude' ), 20, 3 );
			add_filter( 'ywdpd_get_product_ids_involved_adjustment_to_exclude', array( $this, 'return_product_ids_to_exclude' ), 20, 3 );
		}

		/**
		 * Add the vendor taxonomy in counter
		 *
		 * @param array $custom_taxonomy The custom taxonomy.
		 *
		 * @return array
		 * @author YITH
		 * @since 3.0.0
		 */
		public function add_brands_taxonomy_in_counters( $custom_taxonomy ) {

			$custom_taxonomy[] = YITH_WCBR::$brands_taxonomy;

			return $custom_taxonomy;
		}

		/**
		 * Check if the rule is valid
		 *
		 * @param bool             $is_valid If rule is valid.
		 * @param string           $type The rule product type.
		 * @param WC_Product       $product The product.
		 * @param YWDPD_Price_Rule $rule The rule.
		 *
		 * @return bool
		 * @since 3.0.0
		 * @author YITH
		 */
		public function valid_rule_for_brands( $is_valid, $type, $product, $rule ) {

			if ( ! $is_valid ) {
				if ( 'specific_brands' === $type ) {

					$brand_list_ids = $rule->get_meta( 'apply_to_brands_list' );
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
					}
					$is_valid = $rule->is_product_in_list( $product, $brand_list_ids, YITH_WCBR::$brands_taxonomy );
				}
			}

			return $is_valid;
		}

		/**
		 * Check if this product is excluded from a rule
		 *
		 * @param bool             $is_excluded If is excluded.
		 * @param string           $type The rule product type.
		 * @param WC_Product       $product The product.
		 * @param YWDPD_Price_Rule $rule The rule.
		 *
		 * @return bool
		 * @since 3.0.0
		 * @author YITH
		 */
		public function product_excluded_for_brands( $is_excluded, $type, $product, $rule ) {

			if ( 'brand_list_excluded' === $type && $rule->is_exclude_from_apply_enabled() ) {
				$brand_list_ids = $rule->get_meta( 'apply_to_brands_list_excluded' );
				if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
					$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
				}
				$is_excluded = $rule->is_product_in_list( $product, $brand_list_ids, YITH_WCBR::$brands_taxonomy );
			}

			return $is_excluded;
		}

		/**
		 * Get the comulative quantity for brands
		 *
		 * @param int              $quantity The quantity.
		 * @param string           $rule_for The rule type.
		 * @param YWDPD_Price_Rule $rule The rule.
		 *
		 * @return int
		 * @author YITH
		 * @since 3.0.0
		 */
		public function brands_comulative_quantity( $quantity, $rule_for, $rule ) {
			$is_exclude_activated = $rule->is_exclude_from_apply_enabled();
			$type                 = $rule->get_rule_for();

			if ( 'specific_brands' === $type ) {
				$brand_list_ids = $rule->get_meta( 'apply_to_brands_list' );
				if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
					$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
				}
				$quantity = YWDPD_Counter::get_comulative_taxonomy_count( $brand_list_ids, YITH_WCBR::$brands_taxonomy );

			}

			if ( $is_exclude_activated ) {
				$ex_type = $rule->get_exclude_rule_for();
				if ( 'brand_list_excluded' === $ex_type ) {
					$brand_list_ids = $rule->get_meta( 'apply_to_brands_list_excluded' );
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
					}
					$qty_excl = YWDPD_Counter::get_comulative_taxonomy_count( $brand_list_ids, YITH_WCBR::$brands_taxonomy );

					$quantity -= $qty_excl;
				}
			}

			return $quantity;
		}

		/**
		 * Check if the rule is valid to adjustment to
		 *
		 * @param bool             $is_valid Is valid.
		 * @param string           $type The check type.
		 * @param WC_Product       $product The product.
		 * @param YWDPD_Price_Rule $rule The price rule.
		 *
		 * @return bool
		 * @since 3.0.0
		 * @author YITH
		 */
		public function valid_rule_adjustment_to_brands( $is_valid, $type, $product, $rule ) {
			$is_excluded      = false;
			$is_active_adjust = $rule->is_enabled_apply_adjustment_to();
			if ( $is_active_adjust ) {

				if ( $rule->is_exclude_adjustment_to_enabled() && 'brand_list_excluded' === $rule->get_exclude_apply_adjustment_rule_for() ) {
					$brand_list_ids = $rule->get_meta( 'apply_adjustment_brands_list_excluded' );
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
					}
					$is_excluded = $rule->is_product_in_list( $product, $brand_list_ids, YITH_WCBR::$brands_taxonomy );
				}
				if ( ! $is_excluded && 'specific_brands' === $rule->get_rule_apply_adjustment_discount_for() ) {
					$brand_list_ids = $rule->get_meta( 'apply_adjustment_brands_list' );
					if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
						$brand_list_ids = $this->get_translated_brand_ids( $brand_list_ids );
					}
					$is_valid = $rule->is_product_in_list( $product, $brand_list_ids, YITH_WCBR::$brands_taxonomy );
				}
			}

			return $is_valid;
		}

		/**
		 * Add pricing rules options in settings panels
		 *
		 * @param array $rules The cart rule options.
		 *
		 * @return array
		 */
		public function add_cart_rule_option( $rules ) {

			if ( 'ywdpd_cart_rules_product_include_fields' === current_filter() ) {

				$options = array(
					array(
						'id'        => 'enable_require_product_brands',
						'name'      => __( 'Require specific brands in cart', 'ywdpd' ),
						'desc'      => __( 'Enable to require products of specific brands in cart to apply the discount', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product require_product',
						'class'     => 'ywdpd_enable_require_product_brands',
					),
					array(
						'id'        => 'require_product_brands_list',
						'name'      => __( 'Include a list of brands', 'ywdpd' ),
						'desc'      => __( 'Choose which brands are required in cart to apply the discount', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'    => YITH_WCBR::$brands_taxonomy,
							'placeholder' => __( 'Search for brand', 'ywdpd' ),
						),
						'class_row' => 'product require_product enable_require_product_brands_list',
						'multiple'  => true,
					),
					array(
						'id'        => 'require_product_brand_list_mode',
						'name'      => __( 'Apply the discount if:', 'ywdpd' ),
						'desc'      => __( 'Choose whether to apply the discount when at least one of the specified product brand is in the cart or only when all products are in the cart', 'ywdpd' ),
						'type'      => 'radio',
						'options'   => array(
							'at_least'  => __( 'At least one selected brand is in cart', 'ywdpd' ),
							'all_brand' => __( 'All selected brands are in cart', 'ywdpd' ),
						),
						'default'   => 'at_least',
						'class_row' => 'product require_product enable_require_product_brands_list',
						'class'     => 'ywdpd_require_product_tag_list_mode',
					),
				);
				$rules   = array_merge( $rules, $options );
			} elseif ( 'ywdpd_cart_rules_product_exclude_fields' === current_filter() ) {
				$options = array(
					array(
						'id'        => 'enable_exclude_product_brands',
						'name'      => __( 'Exclude specific brands from discount validation', 'ywdpd' ),
						'desc'      => __( 'Enable if you want to exclude specific brands to this cart discount validation', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product exclude_product',
						'class'     => 'ywdpd_enable_exclude_product_brands',
					),
					array(
						'id'        => 'exclude_product_brands_list',
						'name'      => __( 'Exclude a list of brands', 'ywdpd' ),
						'desc'      => __( 'Choose which brands to exclude from this cart discount validation', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'    => YITH_WCBR::$brands_taxonomy,
							'placeholder' => __( 'Search for brand', 'ywdpd' ),
						),
						'class_row' => 'product exclude_product enable_exclude_product_brands_list',
						'multiple'  => true,
					),
				);
				$rules   = array_merge( $rules, $options );
			} elseif ( 'ywdpd_cart_rules_product_disable_fields' === current_filter() ) {
				$options = array(
					array(
						'id'        => 'enable_disable_product_brands',
						'name'      => __( 'Disable discount when specific brands are in cart', 'ywdpd' ),
						'desc'      => __( 'Enable to disable the discount if users have specific brands in their cart', 'ywdpd' ),
						'default'   => 'no',
						'type'      => 'onoff',
						'class_row' => 'product disable_product',
						'class'     => 'ywdpd_enable_disable_product_brands',
					),
					array(
						'id'        => 'disable_product_brands_list',
						'name'      => __( 'Select a list of brands', 'ywdpd' ),
						'desc'      => __( 'Choose which brands will disable the discount', 'ywdpd' ),
						'default'   => '',
						'type'      => 'ajax-terms',
						'data'      => array(
							'taxonomy'    => YITH_WCBR::$brands_taxonomy,
							'placeholder' => __( 'Search for brand', 'ywdpd' ),
						),
						'class_row' => 'product disable_product enable_disable_product_brands_list',
						'multiple'  => true,
					),
				);
				$rules   = array_merge( $rules, $options );
			}

			return $rules;
		}

		/**
		 * Valid the cart rule for vendors
		 *
		 * @param bool            $is_valid If rule is valid.
		 * @param array           $condition The specific condition.
		 * @param array           $conditions All conditions.
		 * @param YWDPD_Cart_Rule $rule The cart rule.
		 *
		 * @return bool
		 */
		public function valid_product_cart_condition( $is_valid, $condition, $conditions, $rule ) {

			$type_check = $rule->get_condition_product_type( $condition );

			if ( ! $is_valid ) {
				if ( 'require_product' === $type_check ) {
					$require_product_brands = yith_plugin_fw_is_true( $condition['enable_require_product_brands'] );
					$brands_to_check        = isset( $condition['require_product_brands_list'] ) ? $condition['require_product_brands_list'] : array();
					$brand_check_mode       = $condition['require_product_brand_list_mode'];
					if ( $require_product_brands && is_array( $brands_to_check ) && count( $brands_to_check ) > 0 ) {
						if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
							$brands_to_check = $this->get_translated_brand_ids( $brands_to_check );
						}
						$is_valid = $rule->check_taxonomy_in_cart( $brands_to_check, YITH_WCBR::$brands_taxonomy, $brand_check_mode );
					}
				} elseif ( 'disable_product' === $type_check ) {
					$disable_brand     = isset( $condition['enable_disable_product_brands'] ) ? yith_plugin_fw_is_true( $condition['enable_disable_product_brands'] ) : false;
					$brands_to_disable = isset( $condition['disable_product_brands_list'] ) ? $condition['disable_product_brands_list'] : array();

					if ( $disable_brand && count( $brands_to_disable ) > 0 ) {
						if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
							$brands_to_disable = $this->get_translated_brand_ids( $brands_to_disable );
						}

						$is_valid = ! $rule->check_taxonomy_in_cart( $brands_to_disable, YITH_WCBR::$brands_taxonomy, 'at_least' );
					}
				}
			}

			return $is_valid;
		}


		/**
		 * Check if brands are  excluded
		 *
		 * @param bool       $is_excluded If the rule is excluded.
		 * @param array      $condition The condition.
		 * @param WC_Product $product The product.
		 *
		 * @return bool
		 */
		public function valid_product_exclude_cart_condition( $is_excluded, $condition, $product ) {

			$exclude_product_brand = isset( $condition['enable_exclude_product_brands'] ) ? $condition['enable_exclude_product_brands'] : false;
			$brands_excluded_list  = isset( $condition['exclude_product_brands_list'] ) ? $condition['exclude_product_brands_list'] : array();
			if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
				$brands_excluded_list = $this->get_translated_brand_ids( $brands_excluded_list );
			}
			if ( ! $is_excluded ) {
				if ( yith_plugin_fw_is_true( $exclude_product_brand ) && count( $brands_excluded_list ) > 0 ) {

					$product_terms = YWDPD_Taxonomy::get_product_term_ids( $product, YITH_WCBR::$brands_taxonomy );

					$is_excluded = count( array_intersect( $product_terms, $brands_excluded_list ) ) > 0;

				}
			}

			return $is_excluded;
		}


		/**
		 * Add pricing rules options in settings panels
		 *
		 * @param array $rules The pricing rule options.
		 *
		 * @return array
		 */
		public function add_pricing_rule_option_rule_for( $rules ) {
			$new_rule = array();
			foreach ( $rules as $key => $rule ) {
				$new_rule[ $key ] = $rule;

				if ( 'rule_for' === $key || 'rule_apply_adjustment_discount_for' === $key ) {
					$new_rule[ $key ]['specific_brands'] = __( 'Specific product brands', 'ywdpd' );

				}

				if ( 'exclude_rule_for' === $key || 'exclude_apply_adjustment_rule_for' === $key ) {
					$new_rule[ $key ]['brand_list_excluded'] = __( 'Specific product brands', 'ywdpd' );
				}
			}

			return $new_rule;
		}

		/**
		 * Add brand search field in the price rule metabox
		 *
		 * @return array
		 * @since 2.0.0
		 * @author YITH
		 */
		public function add_pricing_rule_option() {
			$current_filter = current_filter();
			if ( 'ywdpd_brand_options_rule_for' === $current_filter ) {
				$option = array(
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => YITH_WCBR::$brands_taxonomy,
						'placeholder' => __( 'Search for a brand', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_rule_for',
									'value' => 'specific_brands',
								),
								array(
									'id'    => '_discount_mode',
									'value' => 'bulk,special_offer,gift_products,bogo',
								),
							)
						),
					),
					'label'    => __( 'Apply rule to:', 'ywdpd' ),
					'desc'     => __( 'Search the brand(s) to include in the rule', 'ywdpd' ),
					'multiple' => true,
				);
			} elseif ( 'ywdpd_brand_options_exclude_rule_for' === $current_filter ) {
				$option = array(
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => YITH_WCBR::$brands_taxonomy,
						'placeholder' => __( 'Search for a brand', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_active_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_exclude_rule_for',
									'value' => 'brand_list_excluded',
								),
							)
						),
					),
					'label'    => __( 'Choose which brand(s) to exclude', 'ywdpd' ),
					'desc'     => __( 'Search the brand(s) to exclude from the rule', 'ywdpd' ),
					'multiple' => true,
				);
			} elseif ( 'ywdpd_brand_apply_adjustment_rule_for' === $current_filter ) {
				$option = array(
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => YITH_WCBR::$brands_taxonomy,
						'placeholder' => __( 'Search for a brand', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_rule_apply_adjustment_discount_for',
									'value' => 'specific_brands',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
								array(
									'id'    => '_discount_mode',
									'value' => 'bulk,special_offer',
								),
							)
						),
					),
					'label'    => __( 'Choose which brand(s) to include', 'ywdpd' ),
					'desc'     => __( 'Search the brand(s) to include in this discount', 'ywdpd' ),
					'multiple' => true,
				);
			} else {
				$option = array(
					'type'     => 'ajax-terms',
					'data'     => array(
						'taxonomy'    => YITH_WCBR::$brands_taxonomy,
						'placeholder' => __( 'Search for a brand', 'ywdpd' ),
						'ywdpd-deps'  => wp_json_encode(
							array(
								array(
									'id'    => '_exclude_apply_adjustment_rule_for',
									'value' => 'brand_list_excluded',
								),
								array(
									'id'    => '_active_apply_adjustment_to_exclude',
									'value' => 'yes',
								),
								array(
									'id'      => '_rule_apply_adjustment_discount_for',
									'value'   => 'specific_products',
									'compare' => '!=',
								),
								array(
									'id'    => '_active_apply_discount_to',
									'value' => 'yes',
								),
								array(
									'id'    => '_discount_mode',
									'value' => 'bulk,special_offer',
								),
							)
						),
					),
					'label'    => __( 'Choose which brand(s) to exclude', 'ywdpd' ),
					'desc'     => __( 'Search the brand(s) to exclude from this discount', 'ywdpd' ),
					'multiple' => true,
				);
			}

			return $option;

		}

		/**
		 * Add the brands option in pricing rule metabox
		 *
		 * @param array $pricing_options The options.
		 *
		 * @return array
		 */
		public function add_brands_pricing_options( $pricing_options ) {

			$start        = $pricing_options['tabs']['settings']['fields'];
			$position     = array_search( 'apply_to_tags_list_excluded', array_keys( $start ), true );
			$begin        = array_slice( $start, 0, $position + 1 );
			$end          = array_slice( $start, $position );
			$brands_items = array(
				'apply_to_brands_list'          => array(
					'label'       => __( 'Search for a brand', 'ywdpd' ),
					'type'        => 'brands',
					'desc'        => '',
					'placeholder' => __( 'Search for a brand', 'ywdpd' ),
					'deps'        => array(
						'ids'    => '_apply_to',
						'values' => 'brand_list',
					),
				),
				'apply_to_brands_list_excluded' => array(
					'label'       => __( 'Search for a brand', 'ywdpd' ),
					'type'        => 'brands',
					'desc'        => '',
					'placeholder' => __( 'Search for a branch', 'ywdpd' ),
					'deps'        => array(
						'ids'    => '_apply_to',
						'values' => 'brand_list_excluded',
					),
				),
			);

			$start        = $begin + $brands_items + $end;
			$position     = array_search( 'apply_adjustment_tags_list', array_keys( $start ), true );
			$begin        = array_slice( $start, 0, $position + 1 );
			$end          = array_slice( $start, $position );
			$brands_items = array(
				'apply_adjustment_brands_list'          => array(
					'label'       => __( 'Search for a brand', 'ywdpd' ),
					'type'        => 'brands',
					'desc'        => '',
					'placeholder' => __( 'Search for a brand', 'ywdpd' ),
					'deps'        => array(
						'ids'    => '_apply_adjustment',
						'values' => 'brand_list',
					),
				),
				'apply_adjustment_brands_list_excluded' => array(
					'label'       => __( 'Search for a brand', 'ywdpd' ),
					'type'        => 'brands',
					'desc'        => '',
					'placeholder' => __( 'Search for a branch', 'ywdpd' ),
					'deps'        => array(
						'ids'    => '_apply_adjustment',
						'values' => 'brand_list_excluded',
					),
				),
			);

			$pricing_options['tabs']['settings']['fields'] = $begin + $brands_items + $end;

			return $pricing_options;

		}

		/**
		 * Add the brand taxonomy in popup.
		 *
		 * @param array            $items The args.
		 * @param YWDPD_Price_Rule $price_rule The price rule.
		 * @param string           $rule_for The rule for.
		 *
		 * @return array
		 */
		public function add_brands_args_in_item( $items, $price_rule, $rule_for ) {

			if ( 'specific_brands' === $rule_for ) {
				$items = array(
					'item_ids' => $price_rule->get_meta( 'apply_to_brands_list' ),
					'type'     => YITH_WCBR::$brands_taxonomy,
				);
			}

			return $items;
		}

		/**
		 * Add the brands taxonomy in popup.
		 *
		 * @param array            $items The args.
		 * @param YWDPD_Price_Rule $price_rule The price rule.
		 * @param string           $rule_for The rule for.
		 *
		 * @return array
		 */
		public function add_brands_adjustment_args_in_item( $items, $price_rule, $rule_for ) {

			if ( 'specific_brands' === $rule_for ) {
				$items = array(
					'item_ids' => $price_rule->get_meta( 'apply_adjustment_brands_list' ),
					'type'     => YITH_WCBR::$brands_taxonomy,
				);
			}

			return $items;
		}

		/**
		 * Return the product ids for specific brands
		 *
		 * @param array            $product_ids Product ids.
		 * @param YWDPD_Price_Rule $rule The rule.
		 * @param string           $exclude_for The type of exclusion.
		 *
		 * @return array
		 * @author YITH
		 * @since 2.0.0
		 */
		public function return_product_ids_to_exclude( $product_ids, $rule, $exclude_for ) {

			if ( 'brand_list_excluded' === $exclude_for ) {
				if ( 'ywdpd_get_product_ids_involved_to_exclude' === current_filter() ) {
					$brand_ids = $rule->get_meta( 'apply_to_brands_list_excluded' );
				} else {
					$brand_ids = $rule->get_meta( 'apply_adjustment_brands_list_excluded' );
				}
				if ( YITH_WC_Dynamic_Options::can_wpml_extend_to_translated_object() ) {
					$brand_ids = $this->get_translated_brand_ids( $brand_ids );
				}
				if ( is_array( $brand_ids ) && count( $brand_ids ) > 0 ) {
					$tax_query = WC()->query->get_tax_query(
						array(
							array(
								'taxonomy' => YITH_WCBR::$brands_taxonomy,
								'terms'    => array_values( $brand_ids ),
								'operator' => 'IN',
							),
						)
					);

					$product_ids = get_posts(
						array(
							'numberposts' => - 1,
							'post_type'   => array( 'product' ),
							'post_status' => 'publish',
							'tax_query'   => $tax_query, // phpcs:ignore WordPress.DB.SlowDBQuery
							'fields'      => 'ids',
						)
					);
				}
			}

			return $product_ids;
		}

		/**
		 * Add in the array the translated brand ids.
		 *
		 * @param array $brand_ids The original brand ids.
		 *
		 * @return array
		 * @author YITH
		 * @since 3.0.0
		 */
		public function get_translated_brand_ids( $brand_ids ) {
			$translated_ids = array();

			foreach ( $brand_ids as $brand_id ) {

				$translated_id = intval( yit_wpml_object_id( $brand_id, YITH_WCBR::$brands_taxonomy, true ) );

				if ( intval( $brand_id ) !== $translated_id && ! in_array( $translated_id, $translated_ids, true ) ) {
					$translated_ids[] = $translated_id;
				}
			}

			return count( $translated_ids ) > 0 ? $translated_ids : $brand_ids;
		}
	}
}
