<?php
/**
 * The Quantity table class
 *
 * @package YITH WooCommerce Dynamic Pricing and Discounts\Classes\PriceRules
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The Quantity Table Class.
 */
class YWDPD_Quantity_Table extends YWDPD_Price_Rule {

	/**
	 * Extra data for this object. Name value pairs (name + default value).
	 * Used as a standard way for sub classes (like product types) to add
	 * additional information to an inherited class.
	 *
	 * @since 3.0.0
	 * @var array
	 */
	protected $extra_data = array(
		'rules'            => array(),
		'show_table_price' => 'no',
		'show_in_loop'     => 'yes',
		'table_note'       => '',
	);

	/**
	 * This array will contain the special meta key , the key without _ prefix.
	 *
	 * @since 3.0.0
	 * @var array
	 */
	protected $special_meta = array(
		'rules',
	);

	/**
	 * Get the rule if the ID is passed, otherwise the rule is new and empty.
	 *
	 * @param int|YWDPD_Quantity_Table|object $obj ID to load from the DB (optional) or already queried data.
	 *
	 * @throws Exception The exception.
	 */
	public function __construct( $obj = 0 ) {

		parent::__construct( $obj );

		$this->read();
	}

	/**
	 * Set the quantity rule field.
	 *
	 * @param array $rules the rules.
	 *
	 * @since 3.0.0
	 */
	public function set_rules( $rules ) {
		$this->set_prop( 'rules', $rules );
	}

	/**
	 * Set if show or not the quantity table.
	 *
	 * @param string $show_table Yes or no.
	 *
	 * @since 3.0.0
	 */
	public function set_show_table_price( $show_table ) {
		$this->set_prop( 'show_table_price', $show_table );
	}

	/**
	 * Set if show the discount on loop.
	 *
	 * @param string $show_in_loop Yes or not.
	 *
	 * @since 3.0.0
	 */
	public function set_show_in_loop( $show_in_loop ) {
		$this->set_prop( 'show_in_loop', $show_in_loop );
	}

	/**
	 * Set the table notes
	 *
	 * @param string $note The table notes.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function set_table_note( $note ) {
		$this->set_prop( 'table_note', $note );
	}

	/**
	 * Get the quantity rule fields.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return array
	 * @since 3.0.0
	 */
	public function get_rules( $context = 'view' ) {
		return $this->get_prop( 'rules', $context );
	}

	/**
	 * Get if show or not the quantity table.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 * @since 3.0.0
	 */
	public function get_show_table_price( $context = 'view' ) {
		return $this->get_prop( 'show_table_price', $context );
	}

	/**
	 * Get if show the discount on loop.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 * @since 3.0.0
	 */
	public function get_show_in_loop( $context = 'view' ) {
		return $this->get_prop( 'show_in_loop', $context );
	}

	/**
	 * Get the table notes
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_table_note( $context = 'view' ) {
		return $this->get_prop( 'table_note', $context );
	}

	/**
	 * Check if the table can be show on product.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return bool
	 * @since 3.0.0
	 */
	public function can_show_table_price( $context = 'view' ) {

		$value = yith_plugin_fw_is_true( $this->get_show_table_price( $context ) );

		if ( 'view' === $context ) {
			$value = apply_filters( 'ywdpd_can_show_table_price', $value, $this );
		}

		return $value;
	}

	/**
	 * Check if the discount can show on loop.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return bool
	 * @since 3.0.0
	 */
	public function can_show_discount_in_loop( $context = 'view' ) {

		$value = yith_plugin_fw_is_true( $this->get_show_in_loop( $context ) );

		if ( 'view' === $context ) {
			$value = apply_filters( 'ywdpd_can_show_discount_in_loop', $value, $this );
		}

		return $value;
	}


	/**
	 * Return the discount amount
	 *
	 * @param float $price_to_discount The original price.
	 * @param array $discount_type_rule The discount info.
	 *
	 * @return float
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_discount_amount( $price_to_discount, $discount_type_rule ) {

		$discount_type   = $discount_type_rule['type_discount'];
		$discount_amount = floatval( str_replace( ',', '.', $discount_type_rule['discount_amount'] ) );

		switch ( $discount_type ) {
			case 'percentage':
				$percent          = floatval( $discount_amount / 100 );
				$discounted_price = $price_to_discount - ( $price_to_discount * $percent );
				break;
			case 'price':
				$discount_amount  = apply_filters( 'ywdpd_maybe_should_be_converted', $discount_amount );
				$discounted_price = $price_to_discount - $discount_amount;
				break;
			default:
				$discounted_price = apply_filters( 'ywdpd_maybe_should_be_converted', $discount_amount );
				break;
		}

		if ( apply_filters( 'ywdpd_round_total_price', true ) ) {
			$discounted_price = round( $discounted_price, wc_get_price_decimals() );
		}

		return $discounted_price;
	}

	/**
	 * Return the new price for the specific quantity
	 *
	 * @param float $price_to_discount The price to discount.
	 * @param int $quantity The quantity.
	 * @param WC_Product $product The product object.
	 *
	 * @return float
	 * @since 3.0.0
	 */
	public function get_discounted_price( $price_to_discount, $quantity, $product ) {

		$discounted_price = $price_to_discount;
		$price_rules      = $this->get_rules();

		if ( $price_to_discount ) {
			foreach ( $price_rules as $price_rule ) {
				$min = ! empty( $price_rule['min_quantity'] ) ? $price_rule['min_quantity'] : 1;
				$max = ! empty( $price_rule['max_quantity'] ) ? $price_rule['max_quantity'] : '*';

				if ( ( $quantity >= $min && '*' === $max ) || ( $quantity <= $price_rule['max_quantity'] && $quantity >= $price_rule['min_quantity'] ) ) {

					$discounted_price = $this->get_discount_amount( $price_to_discount, $price_rule );
					break;
				}
			}
		}

		return $discounted_price > 0 ? $discounted_price : 0;
	}

	/**
	 * Add the rule in the cart
	 *
	 * @param string $cart_item_key_to_apply The item key that allow the apply.
	 * @param string $cart_item_key_to_adjust the cart The item key where add the rule.
	 *
	 * @return bool
	 *
	 * @since 3.0.0
	 */
	public function apply_rule_in_cart( $cart_item_key_to_apply, $cart_item_key_to_adjust ) {
		$cart_item_adj = isset( WC()->cart->cart_contents[ $cart_item_key_to_adjust ] ) ? WC()->cart->cart_contents[ $cart_item_key_to_adjust ] : false;

		if ( $cart_item_adj ) {
			$has_bulk_rule = isset( WC()->cart->cart_contents[ $cart_item_key_to_adjust ]['has_bulk_applied'] );
			if ( ! $has_bulk_rule ) {
				$price_to_discount = $this->get_price_to_discount( $cart_item_adj, $cart_item_key_to_adjust );
				$cart_item_apply   = WC()->cart->cart_contents[ $cart_item_key_to_apply ];
				$quantity          = $this->get_quantity( $cart_item_apply );

				$discounted_price = $this->get_discounted_price( $price_to_discount, $quantity, $cart_item_apply['data'] );

				$result = $this->save_discount_in_cart( $cart_item_key_to_adjust, $price_to_discount, $discounted_price );
				if ( $result ) {

					WC()->cart->cart_contents[ $cart_item_key_to_adjust ]['has_bulk_applied'] = true;
				}

				return $result;
			}
		}

		return false;
	}

}

