<?php
/**
 * The Whole discount class
 *
 * @package YITH WooCommerce Dynamic Pricing and Discounts\Classes\Price Rules
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The Whole discount Table Class.
 */
class YWDPD_Discount_Whole extends YWDPD_Price_Rule {

	/**
	 * Extra data for this object. Name value pairs (name + default value).
	 * Used as a standard way for sub classes (like product types) to add
	 * additional information to an inherited class.
	 *
	 * @since 3.0.0
	 * @var array
	 */
	protected $extra_data = array(
		'simple_whole_discount' => array(),
		'show_in_loop'          => 'yes',
	);


	/**
	 * Get the rule if the ID is passed, otherwise the rule is new and empty.
	 *
	 * @param int|YWDPD_Discount_Whole|object $obj ID to load from the DB (optional) or already queried data.
	 *
	 * @throws Exception The exception.
	 */
	public function __construct( $obj = 0 ) {

		parent::__construct( $obj );

		$this->read();
	}

	/**
	 * Set the rule field.
	 *
	 * @param array $rule the rule.
	 *
	 * @since 3.0.0
	 */
	public function set_simple_whole_discount( $rule ) {
		$this->set_prop( 'simple_whole_discount', $rule );
	}

	/**
	 * Set if show the discount on loop.
	 *
	 * @param string $show_in_loop Yes or not.
	 *
	 * @since 3.0.0
	 */
	public function set_show_in_loop( $show_in_loop ) {
		$this->set_prop( 'show_in_loop', $show_in_loop );
	}

	/**
	 * Get if show the discount on loop.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 * @since 3.0.0
	 */
	public function get_show_in_loop( $context = 'view' ) {
		return $this->get_prop( 'show_in_loop', $context );
	}

	/**
	 * Get the quantity rule fields.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return array
	 * @since 3.0.0
	 */
	public function get_simple_whole_discount( $context = 'view' ) {
		return $this->get_prop( 'simple_whole_discount', $context );
	}

	/**
	 * Get if the rule is valid for all products or not.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 * @since 3.0.0
	 */
	public function get_rule_for( $context = 'view' ) {
		return 'all_products';
	}

	/**
	 * Check if the discount can show on loop.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return bool
	 * @since 3.0.0
	 */
	public function can_show_discount_in_loop( $context = 'view' ) {

		$value = yith_plugin_fw_is_true( $this->get_show_in_loop( $context ) );

		if ( 'view' === $context ) {
			$value = apply_filters( 'ywdpd_can_show_discount_in_loop', $value, $this );
		}

		return $value;
	}

	/**
	 * Check if the rule is valid for apply discount.
	 *
	 * @param WC_Product $product The product object.
	 * @param bool       $check_also_variation Check also the variation or not.
	 *
	 * @return bool
	 * @since 3.0.0
	 */
	public function is_valid_to_apply( $product, $check_also_variation = false ) {
		$is_valid = ! $this->is_product_excluded_from_apply( $product, $check_also_variation );

		return apply_filters( 'ywdpd_is_valid_to_apply', $is_valid, 'all_products', $product, $this );
	}

	/**
	 * Check if the rule is valid for adjustment discount.
	 *
	 * @param WC_Product $product The product object.
	 * @param bool       $check_also_variation Check also the variation or not.
	 *
	 * @return bool
	 * @since 3.0.0
	 */
	public function is_valid_to_adjust( $product, $check_also_variation = false ) {
		return false;
	}

	/**
	 * Check if this rule is enabled the adjustment to mode
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return bool
	 * @since 3.0.0
	 */
	public function is_enabled_apply_adjustment_to( $context = 'view' ) {
		return false;
	}

	/**
	 * Return the quantity to check
	 *
	 * @param array $cart_item The cart item.
	 *
	 * @return int;
	 * @since 3.0.0
	 */
	public function get_quantity( $cart_item ) {
		$q = intval( $cart_item['quantity'] );

		return $q;
	}

	/**
	 * Return the new price for the specific quantity
	 *
	 * @param float      $price_to_discount The price to discount.
	 * @param int        $quantity The quantity.
	 * @param WC_Product $product The product object.
	 *
	 * @return float
	 * @since 3.0.0
	 */
	public function get_discounted_price( $price_to_discount, $quantity, $product ) {

		$discounted_price = floatval( $price_to_discount );
		$price_rule       = $this->get_simple_whole_discount();

		if ( $price_to_discount && $quantity >= 1 ) {

			$discount_type   = $price_rule['discount_mode'];
			$discount_amount = floatval( str_replace( ',', '.', $price_rule['discount_value'] ) );

			switch ( $discount_type ) {
				case 'percentage':
					$percent          = $discount_amount / 100;
					$discounted_price = $discounted_price - ( $discounted_price * $percent );
					break;
				case 'price':
					$discount_amount  = apply_filters( 'ywdpd_maybe_should_be_converted', $discount_amount );
					$discounted_price = $discounted_price - $discount_amount;
					break;
				default:
					$discounted_price = apply_filters( 'ywdpd_maybe_should_be_converted', $discount_amount );
					break;

			}
		}
		if ( apply_filters( 'ywdpd_round_total_price', true ) ) {
			$discounted_price = round( $discounted_price, wc_get_price_decimals() );
		}

		return $discounted_price > 0 ? $discounted_price : 0;
	}

	/**
	 * Add the rule in the cart
	 *
	 * @param string $cart_item_key_to_apply The item key that allow the apply.
	 * @param string $cart_item_key_to_adjust the cart The item key where add the rule.
	 *
	 * @return bool
	 * @since 3.0.0
	 */
	public function apply_rule_in_cart( $cart_item_key_to_apply, $cart_item_key_to_adjust ) {
		$cart_item_adj = isset( WC()->cart->cart_contents[ $cart_item_key_to_adjust ] ) ? WC()->cart->cart_contents[ $cart_item_key_to_adjust ] : false;
		if ( $cart_item_adj ) {
			$has_bulk_rule = isset( WC()->cart->cart_contents[ $cart_item_key_to_adjust ]['has_bulk_applied'] );
			if ( ! $has_bulk_rule ) {
				$price_to_discount = $this->get_price_to_discount( $cart_item_adj, $cart_item_key_to_adjust );
				$cart_item_apply   = WC()->cart->cart_contents[ $cart_item_key_to_apply ];
				$quantity          = $this->get_quantity( $cart_item_apply );
				$discounted_price  = $this->get_discounted_price( $price_to_discount, $quantity, $cart_item_adj['data'] );
				$result            = $this->save_discount_in_cart( $cart_item_key_to_adjust, $price_to_discount, $discounted_price );
				if ( $result ) {
					WC()->cart->cart_contents[ $cart_item_key_to_adjust ]['has_bulk_applied'] = true;
				}

				return $result;
			}
		}

		return false;
	}
}

