<?php
/**
 * The Cart rule class
 *
 * @package YITH\DynamicPricingAndDiscounts\Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The cart rule class
 */
class YWDPD_Cart_Rule extends YWDPD_Rule {

	/**
	 * Extra data for this object. Name value pairs (name + default value).
	 * Used as a standard way for sub classes (like product types) to add
	 * additional information to an inherited class.
	 *
	 * @since 3.0.0
	 * @var array
	 */
	protected $extra_data = array(
		'discount_combined'   => 'no',
		'allow_free_shipping' => 'no',
		'cart_discount_rules' => array(),
		'discount_rule'       => array(),
		'show_notice_cart'    => 'no',
		'cart_notice'         => '',
		'cart_notice_bg'      => '',
	);

	/**
	 * Get the rule if the ID is passed, otherwise the rule is new and empty.
	 *
	 * @param int|YWDPD_Cart_Rule|object $obj ID to load from the DB (optional) or already queried data.
	 *
	 * @throws Exception The exception.
	 */
	public function __construct( $obj = 0 ) {
		parent::__construct( $obj );
		$this->read();
		$this->set_internal_meta();
	}

	/**
	 * Set special information about cart rule,useful to show the notices
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	protected function set_internal_meta() {

		$conditions                                     = $this->get_conditions();
		$this->meta_data['has_subtotal_condition']      = false;
		$this->meta_data['invalid_subtotal_conditions'] = array();
		foreach ( $conditions as $condition ) {

			$type = $this->get_condition_type( $condition );
			if ( 'cart_subtotal' === $type ) {
				if ( ! $this->meta_data['has_subtotal_condition'] ) {
					$this->meta_data['has_subtotal_condition'] = true;
				}
				if ( ! $this->is_valid_cart_subtotal_cart_condition( $condition ) ) {
					$this->meta_data['invalid_subtotal_conditions'][] = $condition;
				}
			}
		}
	}

	/**
	 * Set if this rule can be used with other  coupons
	 *
	 * @param string $discount_combined The value ( yes or not ).
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function set_discount_combined( $discount_combined ) {
		$this->set_prop( 'discount_combined', $discount_combined );
	}

	/**
	 * Set if this rule can be allow free shipping
	 *
	 * @param string $allow_free_shipping The value ( yes or not ).
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function set_allow_free_shipping( $allow_free_shipping ) {
		$this->set_prop( 'allow_free_shipping', $allow_free_shipping );
	}

	/**
	 * Set  the discount conditions on this rule
	 *
	 * @param string $cart_discount_rules The conditions.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function set_cart_discount_rules( $cart_discount_rules ) {
		$this->set_prop( 'cart_discount_rules', $cart_discount_rules );
	}

	/**
	 * Set  the discount mode on this rule
	 *
	 * @param array $discount_rule The discount mode.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function set_discount_rule( $discount_rule ) {
		$this->set_prop( 'discount_rule', $discount_rule );
	}

	/**
	 * Set if show or not the notice in the cart
	 *
	 * @param string $show_notice Yes or no.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function set_show_notice_cart( $show_notice ) {
		$this->set_prop( 'show_notice_cart', $show_notice );
	}

	/**
	 * Set the notice to show
	 *
	 * @param string $cart_notice The notice.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function set_cart_notice( $cart_notice ) {
		$this->set_prop( 'cart_notice', $cart_notice );
	}

	/**
	 * Set the background color for the notice.
	 *
	 * @param string $bg_color An exadecimal color.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function set_cart_notice_bg( $bg_color ) {
		$this->set_prop( 'cart_notice_bg', $bg_color );
	}

	/**
	 * Get if this rule can be used with other  coupons
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_discount_combined( $context = 'view' ) {
		return $this->get_prop( 'discount_combined', $context );
	}

	/**
	 * Get if this rule can be allow free shipping
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_allow_free_shipping( $context = 'view' ) {
		return $this->get_prop( 'allow_free_shipping', $context );
	}

	/**
	 * Get the discount conditions on this rule
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_cart_discount_rules( $context = 'view' ) {
		return $this->get_prop( 'cart_discount_rules', $context );
	}

	/**
	 * Get  the discount mode on this rule
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return array
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_discount_rule( $context = 'view' ) {
		return $this->get_prop( 'discount_rule', $context );
	}

	/**
	 * Get if show or not the notice in the cart
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_show_notice_cart( $context ) {
		return $this->get_prop( 'show_notice_cart', $context );
	}

	/**
	 * Get the notice to show
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_cart_notice( $context = 'view' ) {
		return $this->get_prop( 'cart_notice', $context );
	}

	/**
	 * Get the background color for the notice.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return string
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_cart_notice_bg( $context = 'view' ) {
		return $this->get_prop( 'cart_notice_bg', $context );
	}

	/**
	 * Check if the rule can be used also with other coupons
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return bool
	 * @since 3.0.0
	 * @author YITH
	 */
	public function can_be_used_with_other_coupons( $context = 'view' ) {
		$disable = $this->get_discount_combined( $context );

		return ! yith_plugin_fw_is_true( $disable );
	}

	/**
	 * Check if the rule allow the free shipping or not.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return bool
	 * @since 3.0.0
	 * @author YITH
	 */
	public function allow_free_shipping( $context = 'view' ) {

		$allow = $this->get_allow_free_shipping( $context );

		return yith_plugin_fw_is_true( $allow );
	}

	/**
	 * Get all rule conditions
	 *
	 * @return array
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_conditions() {
		$discount_rules = $this->get_cart_discount_rules();

		return maybe_unserialize( $discount_rules );
	}

	/**
	 * Get the discount type of this rule
	 *
	 * @return string
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_discount_type() {
		$discounts = $this->get_discount_rule();
		$type      = isset( $discounts['discount_type'] ) ? $discounts['discount_type'] : 'percentage';

		return $type;
	}

	/**
	 * Get the discount amount of this rule
	 *
	 * @return float
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_discount_amount() {
		$discounts = $this->get_discount_rule();
		$amount    = isset( $discounts['discount_amount'] ) ? wc_format_decimal( $discounts['discount_amount'], wc_get_price_decimals() ) : 10;

		return $amount;
	}

	/**
	 * Get the name of current condition
	 *
	 * @param array $conditon The condition.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_condition_name( $conditon ) {

		$conditon_name = isset( $conditon['cart_condition_name'] ) ? $conditon['cart_condition_name'] : '';

		return $conditon_name;
	}

	/**
	 * Get the type of current condition
	 *
	 * @param array $conditon The condition.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_condition_type( $conditon ) {

		$conditon_name = isset( $conditon['condition_for'] ) ? $conditon['condition_for'] : 'customers';

		return $conditon_name;
	}

	// User condition methods.

	/**
	 * Return how the condition is applied for users ( All | Specific users )
	 *
	 * @param array $condition The single condition.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_condition_user_type( $condition ) {
		$user_apply_to = isset( $condition['user_discount_to'] ) ? $condition['user_discount_to'] : 'all';

		return $user_apply_to;
	}

	/**
	 * Check if is enabled the users exclusion
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_enabled_user_exclusion( $condition ) {
		$is_enabled = isset( $condition['enable_exclude_users'] ) ? $condition['enable_exclude_users'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the list of customers excluded
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_customer_list_excluded( $condition ) {

		$list = ! empty( $condition['customers_list_excluded'] ) ? $condition['customers_list_excluded'] : array();

		return $list;
	}

	/**
	 * Get the list of user role excluded
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_user_role_list_excluded( $condition ) {

		$list = ! empty( $condition['customers_role_list_excluded'] ) ? $condition['customers_role_list_excluded'] : array();

		return $list;
	}

	/**
	 * Get the list of customers included
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_customer_list_included( $condition ) {

		$list = ! empty( $condition['customers_list'] ) ? $condition['customers_list'] : array();

		return $list;
	}

	/**
	 * Get the list of user roles included
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_user_role_list_included( $condition ) {

		$list = ! empty( $condition['customers_role_list'] ) ? $condition['customers_role_list'] : array();

		return $list;
	}

	// Number of orders condition.

	/**
	 * Get the minimum number of orders
	 *
	 * @param array $condition The single condition.
	 *
	 * @return int
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_minimum_orders( $condition ) {
		$value = isset( $condition['min_order'] ) ? $condition['min_order'] : 1;

		return $value;
	}

	/**
	 * Get the maximum number of orders
	 *
	 * @param array $condition The single condition.
	 *
	 * @return int
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_maximum_orders( $condition ) {
		$value = ! empty( $condition['max_order'] ) ? $condition['max_order'] : '';

		return $value;
	}

	// Total past expense condition.

	/**
	 * Get the minimum past expense required
	 *
	 * @param array $condition The single condition.
	 *
	 * @return float
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_minimum_past_expense( $condition ) {

		$value = isset( $condition['min_expense'] ) ? wc_format_decimal( $condition['min_expense'], wc_get_price_decimals() ) : 1;

		return $value;
	}

	/**
	 * Get the maximum past expense allowed
	 *
	 * @param array $condition The single condition.
	 *
	 * @return float
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_maximum_past_expense( $condition ) {

		$value = ! empty( $condition['max_expense'] ) ? wc_format_decimal( $condition['max_expense'], wc_get_price_decimals() ) : '';

		return $value;
	}

	// Product condition.

	/**
	 * Get the product type condition ( Require, Exclude, Disable )
	 *
	 * @param array $condition The single condition.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_condition_product_type( $condition ) {
		$type = isset( $condition['product_type'] ) ? $condition['product_type'] : 'require_product';

		return $type;
	}

	/**
	 * Check if specific product are required in cart
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_required_specific_product( $condition ) {
		$is_enabled = isset( $condition['enable_require_product'] ) ? $condition['enable_require_product'] : 'yes';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the required product list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_required_product_list( $condition ) {
		$list = ! empty( $condition['require_product_list'] ) ? $condition['require_product_list'] : array();

		return apply_filters( 'ywdpd_cart_rule_get_required_product_list', $list, $this );
	}

	/**
	 * Get how manage the required list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_required_product_list_mode( $condition ) {
		$value = isset( $condition['require_product_list_mode'] ) ? $condition['require_product_list_mode'] : 'at_least';

		return $value;
	}

	/**
	 * Check if specific product categories are required in cart
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_required_specific_product_categories( $condition ) {
		$is_enabled = isset( $condition['enable_require_product_categories'] ) ? $condition['enable_require_product_categories'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the required category product list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_required_product_category_list( $condition ) {
		$list = ! empty( $condition['require_product_category_list'] ) ? $condition['require_product_category_list'] : array();

		return apply_filters( 'ywdpd_cart_rule_get_required_product_category_list', $list, $this );
	}

	/**
	 * Get how manage the required list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_required_product_category_list_mode( $condition ) {
		$value = isset( $condition['require_product_cat_list_mode'] ) ? $condition['require_product_cat_list_mode'] : 'at_least';

		return $value;
	}

	/**
	 * Check if specific product tag are required in cart
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_required_specific_product_tag( $condition ) {
		$is_enabled = isset( $condition['enable_require_product_tag'] ) ? $condition['enable_require_product_tag'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the required tag product list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_required_product_tag_list( $condition ) {
		$list = ! empty( $condition['require_product_tag_list'] ) ? $condition['require_product_tag_list'] : array();

		return apply_filters( 'ywdpd_cart_rule_get_required_product_tag_list', $list, $this );
	}

	/**
	 * Get how manage the required list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_required_product_tag_list_mode( $condition ) {
		$value = isset( $condition['require_product_tag_list_mode'] ) ? $condition['require_product_tag_list_mode'] : 'at_least';

		return $value;
	}

	/**
	 * Check if products are excluded form rule
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_excluded_product_enabled( $condition ) {
		$is_enabled = isset( $condition['enable_exclude_require_product'] ) ? $condition['enable_exclude_require_product'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the excluded product list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_excluded_product_list( $condition ) {
		$list = ! empty( $condition['exclude_product_list'] ) ? $condition['exclude_product_list'] : array();

		return apply_filters( 'ywdpd_cart_rule_get_excluded_product_list', $list, $this );
	}

	/**
	 * Check if on-sale products are excluded form rule
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_onsale_product_excluded( $condition ) {
		$is_enabled = isset( $condition['enable_exclude_on_sale_product'] ) ? $condition['enable_exclude_on_sale_product'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Check if product categories are excluded form rule
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_excluded_product_categories_enabled( $condition ) {
		$is_enabled = isset( $condition['enable_exclude_product_categories'] ) ? $condition['enable_exclude_product_categories'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the excluded product list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_excluded_product_categories_list( $condition ) {
		$list = ! empty( $condition['exclude_product_category_list'] ) ? $condition['exclude_product_category_list'] : array();

		return apply_filters( 'ywdpd_cart_rule_get_excluded_product_categories_list', $list, $this );
	}

	/**
	 * Check if product tag are excluded form rule
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_excluded_product_tag_enabled( $condition ) {
		$is_enabled = isset( $condition['enable_exclude_product_tag'] ) ? $condition['enable_exclude_product_tag'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the excluded product tag list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_excluded_product_tag_list( $condition ) {
		$list = ! empty( $condition['exclude_product_tag_list'] ) ? $condition['exclude_product_tag_list'] : array();

		return apply_filters( 'ywdpd_cart_rule_get_excluded_product_tag_list', $list, $this );
	}

	/**
	 * Check if products are excluded form rule
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_disable_product_enabled( $condition ) {
		$is_enabled = isset( $condition['enable_disable_require_product'] ) ? $condition['enable_disable_require_product'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the product list that disable the rule
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_disable_product_list( $condition ) {
		$list = ! empty( $condition['disable_product_list'] ) ? $condition['disable_product_list'] : array();

		return apply_filters( 'ywdpd_cart_rule_get_disable_product_list', $list, $this );
	}

	/**
	 * Check if product categories are disabled form rule
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_disable_product_categories_enabled( $condition ) {
		$is_enabled = isset( $condition['enable_disable_product_categories'] ) ? $condition['enable_disable_product_categories'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the excluded product list
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_disable_product_categories_list( $condition ) {
		$list = ! empty( $condition['disable_product_category_list'] ) ? $condition['disable_product_category_list'] : array();

		return apply_filters( 'ywdpd_cart_rule_get_disable_product_categories_list', $list, $this );
	}

	/**
	 * Check if product tag are disable form rule
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_disable_product_tag_enabled( $condition ) {
		$is_enabled = isset( $condition['disable_exclude_product_tag'] ) ? $condition['disable_exclude_product_tag'] : 'no';

		return yith_plugin_fw_is_true( $is_enabled );
	}

	/**
	 * Get the product tag list that disable this rule
	 *
	 * @param array $condition The single condition.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_disable_product_tag_list( $condition ) {
		$list = ! empty( $condition['disable_product_tag_list'] ) ? $condition['disable_product_tag_list'] : array();

		return apply_filters( 'ywdpd_cart_rule_get_disable_product_tag_list', $list, $this );
	}

	// CART ITEM condition.

	/**
	 * Get cart item condition type ( Product items or Cart item )
	 *
	 * @param array $condition The single condition.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_cart_item_condition_mode( $condition ) {
		$value = isset( $condition['cart_item_qty_type'] ) ? $condition['cart_item_qty_type'] : 'count_product_items';

		return $value;
	}

	/**
	 * Get the minimum item in cart
	 *
	 * @param array $condition The single condition.
	 *
	 * @return int
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_minimum_product_item( $condition ) {
		$value = isset( $condition['min_product_item'] ) ? intval( $condition['min_product_item'] ) : 1;

		return $value;
	}

	/**
	 * Get the maximum item in cart allowed
	 *
	 * @param array $condition The single condition.
	 *
	 * @return int
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_maximum_product_item( $condition ) {
		$value = ! empty( $condition['max_product_item'] ) ? intval( $condition['max_product_item'] ) : '';

		return $value;
	}

	/**
	 * Get the minimum cart element in cart
	 *
	 * @param array $condition The single condition.
	 *
	 * @return int
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_minimum_cart_item( $condition ) {
		$value = isset( $condition['min_cart_item'] ) ? intval( $condition['min_cart_item'] ) : 1;

		return $value;
	}

	/**
	 * Get the maximum cart item allowed
	 *
	 * @param array $condition The single condition.
	 *
	 * @return int
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_maximum_cart_item( $condition ) {
		$value = ! empty( $condition['max_cart_item'] ) ? intval( $condition['max_cart_item'] ) : '';

		return $value;
	}

	// SUBTOTAL condition.

	/**
	 * Get the minimum subtotal required
	 *
	 * @param array $condition The single condition.
	 *
	 * @return float
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_minimum_subototal( $condition ) {
		$value = isset( $condition['min_subtotal'] ) ? wc_format_decimal( $condition['min_subtotal'], wc_get_price_decimals() ) : 1;

		return apply_filters( 'ywdpd_cart_rule_get_minimum_subtotal', $value, $this );
	}

	/**
	 * Get the maximum subtotal required
	 *
	 * @param array $condition The single condition.
	 *
	 * @return float
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_maximum_subototal( $condition ) {
		$value = ! empty( $condition['max_subtotal'] ) ? wc_format_decimal( $condition['max_subtotal'], wc_get_price_decimals() ) : '';

		return apply_filters( 'ywdpd_cart_rule_get_maximum_subtotal', $value, $this );
	}

	/**
	 * Return the rule type ( Price or Cart )
	 *
	 * @return string
	 * @since 3.0.0
	 */
	public function get_type() {
		return 'cart';
	}

	/**
	 * Check if this rule is Valid.
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return bool
	 * @since 3.0.0
	 */
	public function is_valid( $context = 'view' ) {
		$is_valid = false;

		if ( $this->is_enabled( $context ) && $this->is_scheduled( $context ) ) {
			$conditions = $this->get_conditions();
			$is_valid   = true;
			foreach ( $conditions as $condition ) {

				$function_to_call = 'is_valid_' . $this->get_condition_type( $condition ) . '_cart_condition';
				if ( is_callable( array( $this, $function_to_call ) ) ) {

					$is_valid = $this->$function_to_call( $condition );
				}

				if ( ! $is_valid ) {
					break;
				}
			}
		}

		return $is_valid;
	}

	/**
	 * Check if the current user is valid
	 *
	 * @param array $condition The single conditions.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_valid_customers_cart_condition( $condition ) {
		$is_valid       = true;
		$user_logged_in = is_user_logged_in();
		$user           = $user_logged_in ? wp_get_current_user() : false;
		$user_roles     = $user ? $user->roles : array( 'guest' );
		$user_mode      = $this->get_condition_user_type( $condition );

		if ( 'all' === $user_mode ) {
			$is_customers_excluded = $this->is_enabled_user_exclusion( $condition );
			if ( $is_customers_excluded ) {
				$customers_excluded = $this->get_customer_list_excluded( $condition );
				$roles_excluded     = $this->get_user_role_list_excluded( $condition );
				if ( $user_logged_in ) {
					if ( ! empty( $customers_excluded ) ) {
						if ( in_array( $user->ID, $customers_excluded ) ) { //phpcs:ignore
							$is_valid = false;
						}
					}
				}

				if ( ! empty( $roles_excluded ) && is_array( $roles_excluded ) ) {
					$intersect = array_intersect( $user_roles, $roles_excluded );
					if ( ! empty( $roles_excluded ) && 0 < count( $intersect ) ) {
						$is_valid = false;
					}
				}
			}
		} else {
			$customers_list = $this->get_customer_list_included( $condition );
			$role_list      = $this->get_user_role_list_included( $condition );

			if ( $user_logged_in ) {
				if ( ! empty( $customers_list ) ) {
					if ( ! in_array( $user->ID, $customers_list ) ) { // phpcs:ignore
						$is_valid = false;
					}
				}
			} else {
				$is_valid = false;
			}

			if ( ! empty( $role_list ) && is_array( $role_list ) ) {
				$intersect = array_intersect( $user_roles, $role_list );

				if ( 0 === count( $intersect ) ) {
					$is_valid = false;
				}
			}
		}

		return apply_filters( 'ywdpd_customers_condition_in_cart_is_valid', $is_valid, $condition, $this->get_conditions(), $this );
	}

	/**
	 * Check if customer has the right amount of orders
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_valid_num_of_orders_cart_condition( $condition ) {
		$is_valid = true;

		if ( is_user_logged_in() ) {

			$user_id      = get_current_user_id();
			$num_of_order = wc_get_customer_order_count( $user_id );
			$min_order    = $this->get_minimum_orders( $condition );
			$max_order    = $this->get_maximum_orders( $condition );

			if ( ( $num_of_order < $min_order ) || ( ! empty( $max_order ) && $num_of_order >= $max_order ) ) {

				$is_valid = false;
			}
		}

		return $is_valid;
	}

	/**
	 * Check if the customer is valid for past expense
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_valid_past_expense_cart_condition( $condition ) {
		$is_valid = true;

		if ( is_user_logged_in() ) {

			$user_id      = get_current_user_id();
			$past_expense = wc_get_customer_total_spent( $user_id );
			$min_expense  = $this->get_minimum_past_expense( $condition );
			$max_expense  = $this->get_maximum_past_expense( $condition );

			if ( ( $past_expense < $min_expense ) || ( ! empty( $max_expense ) && $past_expense > $max_expense ) ) {
				$is_valid = false;
			}
		}

		return $is_valid;
	}

	/**
	 * Check if the product condition is valid
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_valid_product_cart_condition( $condition ) {

		$is_valid = true;

		$type_check = $this->get_condition_product_type( $condition );
		if ( 'require_product' === $type_check ) {

			$require_product     = $this->is_required_specific_product( $condition );
			$require_product_cat = $this->is_required_specific_product_categories( $condition );
			$require_product_tag = $this->is_required_specific_product_tag( $condition );
			$is_valid            = false;
			if ( $require_product ) {
				$product_list = $this->get_required_product_list( $condition );
				$check_mode   = $this->get_required_product_list_mode( $condition );

				if ( count( $product_list ) > 0 ) {
					$is_valid = $this->check_taxonomy_in_cart( $product_list, 'product', $check_mode );
				}
			}
			if ( ! $is_valid && $require_product_cat ) {
				$cat_list   = $this->get_required_product_category_list( $condition );
				$check_mode = $this->get_required_product_category_list_mode( $condition );

				$is_valid = $this->check_taxonomy_in_cart( $cat_list, 'product_cat', $check_mode );

			}

			if ( ! $is_valid && $require_product_tag ) {
				$tag_list   = $this->get_required_product_tag_list( $condition );
				$check_mode = $this->get_required_product_list_mode( $condition );
				$is_valid   = $this->check_taxonomy_in_cart( $tag_list, 'product_tag', $check_mode );
			}
		} elseif ( 'disable_product' === $type_check ) {
			$disable_product     = yith_plugin_fw_is_true( $condition['enable_disable_require_product'] );
			$disable_product_cat = yith_plugin_fw_is_true( $condition['enable_disable_product_categories'] );
			$disable_product_tag = yith_plugin_fw_is_true( $condition['disable_exclude_product_tag'] );
			$is_disable          = false;
			if ( $disable_product ) {
				$disable_product_list = $this->get_disable_product_list( $condition );
				$is_disable           = $this->check_taxonomy_in_cart( $disable_product_list, 'product', 'at_least' );
			}

			if ( ! $is_disable && $disable_product_cat ) {
				$disable_product_cat_list = $this->get_disable_product_categories_list( $condition );
				$is_disable               = $this->check_taxonomy_in_cart( $disable_product_cat_list, 'product_cat', 'at_least' );
			}
			if ( ! $is_disable && $disable_product_tag ) {
				$disable_product_tag_list = $this->get_disable_product_list( $condition );
				$is_disable               = $this->check_taxonomy_in_cart( $disable_product_tag_list, 'product_tag', 'at_least' );
			}

			$is_valid = ! $is_disable;
		}

		return apply_filters( 'ywdpd_valid_product_cart_condition', $is_valid, $condition, $this->get_conditions(), $this );

	}

	/**
	 * Check if the product is in a exclusion condition
	 *
	 * @param WC_Product $product The product.
	 *
	 * @return bool
	 * @since 3.0.0
	 * @author YITH
	 */
	public function is_product_excluded_from_conditions( $product ) {
		$is_excluded = false;
		$conditions  = $this->get_conditions();
		foreach ( $conditions as $condition ) {
			$type = $this->get_condition_type( $condition );
			if ( 'product' === $type ) {
				$type_check = $this->get_condition_product_type( $condition );

				if ( 'exclude_product' === $type_check ) {
					$exclude_product     = $this->is_excluded_product_enabled( $condition );
					$exclude_onsale      = $this->is_onsale_product_excluded( $condition );
					$exclude_product_cat = $this->is_excluded_product_categories_enabled( $condition );
					$exclude_product_tag = $this->is_excluded_product_tag_enabled( $condition );

					if ( $exclude_product ) {
						$product_list = $this->get_excluded_product_list( $condition );

						$is_excluded = in_array( $product->get_id(), $product_list ); //phpcs:ignore
					}
					if ( ! $is_excluded && $exclude_onsale && $product->is_on_sale() ) {
						$is_excluded = true;
					}
					if ( ! $is_excluded && $exclude_product_cat ) {
						$exclude_product_cat_list = $this->get_excluded_product_categories_list( $condition );
						$product_terms            = YWDPD_Taxonomy::get_product_category_ids( $product );
						$is_excluded              = count( array_intersect( $product_terms, $exclude_product_cat_list ) ) > 0;
					}

					if ( ! $is_excluded && $exclude_product_tag ) {
						$exclude_product_tag_list = $this->get_excluded_product_tag_list( $condition );
						$product_terms            = YWDPD_Taxonomy::get_product_tag_ids( $product );
						$is_excluded              = count( array_intersect( $product_terms, $exclude_product_tag_list ) ) > 0;
					}

					$is_excluded = apply_filters( 'ywpd_is_excluded_product_cart_condition', $is_excluded, $condition, $product, $this );

					if ( $is_excluded ) {
						break;
					}
				}
			}
		}

		return apply_filters( 'ywdpd_valid_product_exclude_cart_condition', $is_excluded, $conditions, $product, $this );
	}

	/**
	 * Check if the cart item
	 *
	 * @param array $condition The condition.
	 *
	 * @return bool
	 * @since 3.0.0
	 * @author YITH
	 */
	public function is_valid_cart_items_cart_condition( $condition ) {
		$is_valid       = true;
		$quantity_check = $this->get_cart_item_condition_mode( $condition );

		if ( 'count_product_items' === $quantity_check ) {
			$min_items = $this->get_minimum_product_item( $condition );
			$max_items = $this->get_maximum_product_item( $condition );
			$is_valid  = $this->check_cart_item_quantity( $min_items, $max_items );
		} else {
			$min_items = $this->get_minimum_cart_item( $condition );
			$max_items = $this->get_maximum_cart_item( $condition );
			$is_valid  = $this->check_comulative_quantity( $min_items, $max_items );
		}

		return $is_valid;
	}

	/**
	 * Check if the subtotal condition is valid
	 *
	 * @param array $condition The single condition.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_valid_cart_subtotal_cart_condition( $condition ) {
		$is_valid = true;

		$min_subtotal = $this->get_minimum_subototal( $condition );
		$max_subtotal = $this->get_maximum_subototal( $condition );
		$subtotal     = $this->get_cart_subtotal();
		if ( $subtotal < $min_subtotal || ( ! empty( $max_subtotal ) && $subtotal > $max_subtotal ) ) {
			$is_valid = false;
		}

		return $is_valid;
	}

	/**
	 * Check if the ids in the list are in cart
	 *
	 * @param array  $list The ids list to check.
	 * @param string $taxonomy The taxonomy name.
	 * @param string $mode The mode to check.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function check_taxonomy_in_cart( $list, $taxonomy, $mode ) {
		$check = false;

		foreach ( $list as $id ) {
			if ( 'product' === $taxonomy ) {

				if ( 'product_variation' === get_post_type( $id ) ) {
					$check = YWDPD_Counter::get_variation_count( $id ) > 0;
				} else {
					$check = YWDPD_Counter::get_product_count( $id ) > 0;
				}
			} else {
				if ( 'product_cat' === $taxonomy ) {
					$check = YWDPD_Counter::get_category_count( $id ) > 0;
				} elseif ( 'product_tag' === $taxonomy ) {
					$check = YWDPD_Counter::get_tag_count( $id ) > 0;
				} else {
					$check = YWDPD_Counter::get_taxonomy_count( $id, $taxonomy ) > 0;
				}
			}
			if ( 'at_least' === $mode && $check ) {
				break;
			} elseif ( 'all_product' === $mode && ! $check ) {
				break;
			}
		}

		return $check;
	}

	/**
	 * Check if each cart item quantity is valid
	 *
	 * @param int $min_qty The min qty to check.
	 * @param int $max_qty The max qty to check.
	 *
	 * @return bool
	 * @since 3.0
	 * @author YITH
	 */
	public function check_cart_item_quantity( $min_qty, $max_qty ) {
		$is_valid = true;

		if ( ! is_null( WC()->cart ) && ! WC()->cart->is_empty() ) {

			foreach ( WC()->cart->get_cart_contents() as $cart_item_key => $cart_item ) {

				$qty = $cart_item['quantity'];

				if ( apply_filters( 'ywdpd_product_can_be_counted_in_condition', ! $this->is_product_excluded_from_conditions( $cart_item['data'] ), $cart_item ) ) {
					if ( ( $qty < $min_qty ) || ( ! empty( $max_qty ) && $qty > $max_qty ) ) {
						return false;
					}
				}
			}
		}

		return $is_valid;
	}

	/**
	 * Check if the total of items in the cart are between min and max
	 *
	 * @param int $min_items The minimum amount of items in the cart.
	 * @param int $max_items The maximum amount of items in the cart.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function check_comulative_quantity( $min_items, $max_items ) {

		$num_items = 0;

		if ( ! is_null( WC()->cart ) && ! WC()->cart->is_empty() ) {

			foreach ( WC()->cart->get_cart_contents() as $cart_item_key => $cart_item ) {

				if ( apply_filters( 'ywdpd_product_can_be_counted_in_condition', ! $this->is_product_excluded_from_conditions( $cart_item['data'] ), $cart_item ) ) {
					$num_items += $cart_item['quantity'];
				}
			}
		}
		$num_items = apply_filters( 'ywdpd_get_cart_item_quantities', $num_items );

		return ( ( $num_items >= $min_items ) && ( ( empty( $max_items ) || ( ! empty( $max_items ) && $num_items <= $max_items ) ) ) );
	}

	/**
	 * Return the right cart subtotal
	 *
	 * @return float
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_cart_subtotal() {

		$subtotal = 0;
		if ( ! is_null( WC()->cart ) && ! WC()->cart->is_empty() ) {
			$tax_excluded = 'tax_excluded' === YITH_WC_Dynamic_Options::how_calculate_discounts();
			foreach ( WC()->cart->get_cart_contents() as $cart_item_key => $cart_item ) {
				if ( isset( $cart_item['data'], $cart_item['line_subtotal'], $cart_item['line_subtotal_tax'] ) ) {
					if ( apply_filters( 'ywdpd_product_can_be_counted_in_condition', ! $this->is_product_excluded_from_conditions( $cart_item['data'] ), $cart_item ) ) {
						$subtotal += $tax_excluded ? $cart_item['line_subtotal'] : $cart_item['line_subtotal'] + $cart_item['line_subtotal_tax'];
					}
				}
			}
		}

		return $subtotal;
	}


	/**
	 * Check if the notice are enable or not
	 *
	 * @param string $context What the value is for. Valid values are view and edit.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_notice_enabled( $context = 'view' ) {

		$value = yith_plugin_fw_is_true( $this->get_show_notice_cart( $context ) );

		return $value && $this->get_meta( 'has_subtotal_condition' );
	}

	/**
	 * Get all invalid subtotal conditions
	 *
	 * @return array
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_invalid_subtotal_conditions() {
		return $this->get_meta( 'invalid_subtotal_conditions' );
	}
}
