<?php
/**
 * This class manage all frontend features
 *
 * @package YITH\DynamicPricingAndDiscounts\Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Class YWDPD_Frontend
 */
class YWDPD_Frontend {
	/**
	 * The unique access of the class.
	 *
	 * @var YWDPD_Frontend
	 */
	protected static $instance;

	/**
	 * This array will contain all dynamic prices ( the key will be the product id )
	 *
	 * @var array
	 */
	protected $has_get_price_filter;

	/**
	 * This array will contain all dynamic formatted prices ( the key will be the product id )
	 *
	 * @var array
	 */
	protected $has_get_price_html_filter;

	/**
	 * This array will contain for each product the bulk rule applied, each product can have only one bulk rule applied
	 * ( Quantity, Whole discount and Category discount are bulk rule )
	 *
	 * @var array
	 */
	protected $valid_bulk_rule;

	protected $all_bulk_rules;

	/**
	 * YWDPD_Frontend constructor.
	 *
	 * @author YITH
	 * @sice 3.0.0
	 */
	private function __construct() {

		$this->has_get_price_filter      = array();
		$this->has_get_price_html_filter = array();
		$this->valid_bulk_rule           = array();
		$this->all_bulk_rules            = array();

		add_filter( 'woocommerce_get_price_html', array( $this, 'get_product_price_html' ), 10, 2 );
		add_filter( 'woocommerce_variable_price_html', array( $this, 'get_product_variable_price_html' ), 10, 2 );
		add_filter( 'woocommerce_available_variation', array( $this, 'add_quantity_table_to_available_variation' ), 10, 3 );
		add_filter( 'woocommerce_show_variation_price', array( $this, 'show_variation_price' ), 99, 3 );
		add_action( 'ywdpd_before_calculate_discounts', array( $this, 'remove_price_filters' ), 99 );
		add_action( 'ywdpd_after_calculate_discounts', array( $this, 'add_price_filters' ), 99 );
		add_action( 'ywdpd_before_replace_cart_item_price', array( $this, 'remove_price_filters' ), 99 );
		add_action( 'ywdpd_after_replace_cart_item_price', array( $this, 'add_price_filters' ), 99 );
		add_action( 'init', array( $this, 'init_tables_and_notices' ), 20 );
		add_filter( 'woocommerce_cart_item_price', array( $this, 'replace_cart_item_price' ), 100, 3 );
		add_filter( 'woocommerce_cart_item_subtotal', array( $this, 'replace_cart_item_subtotal' ), 100, 3 );

		if ( YITH_WC_Dynamic_Options::show_discount_info_in_cart() ) {
			add_action( 'woocommerce_cart_totals_before_shipping', array( $this, 'show_total_discount_message_on_cart' ), 99 );
		}

		// Store the rules in the order item.
		add_action( 'woocommerce_checkout_create_order_line_item', array( $this, 'save_dynamic_rules_in_order_item' ), 20, 3 );
		add_action( 'woocommerce_checkout_create_order_coupon_item', array( $this, 'save_dynamic_coupon_name' ), 20, 2 );

		add_action( 'woocommerce_order_item_meta_end', array( $this, 'format_custom_meta_data' ), 20, 2 );

		add_action( 'woocommerce_before_cart_table', array( $this, 'show_cart_notices' ) );

	}

	/**
	 * Return the unique instance of the class.
	 *
	 * @return YWDPD_Frontend
	 * @since 3.0.0
	 * @author YITH
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {

			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Prevent the clone of object
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	private function __clone() {
	}

	/**
	 * Prevent unserialize the object.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function __wakeup() {
	}


	/**
	 * Check if is possible calculate the dynamic price.
	 *
	 * @param float      $price The product price.
	 * @param WC_Product $product The product.
	 *
	 * @return bool
	 * @since 3.0.0
	 * @author YITH
	 */
	public function can_calculate_dynamic_price( $price, $product ) {

		$can_calculate = ! ( is_cart() || is_checkout() || empty( $price ) );

		return apply_filters( 'ywdpd_can_calculate_dynamic_price', $can_calculate, $product );
	}

	/**
	 * Return the right price html with the dynamic price format.
	 *
	 * @param string     $price_html The price html.
	 * @param WC_Product $product The product.
	 *
	 * @return string
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_product_price_html( $price_html, $product ) {
		$this->remove_price_filters();
		if ( 'variable' !== $product->get_type() && $this->can_calculate_dynamic_price( $price_html, $product ) ) {
			if ( isset( $this->has_get_price_html_filter[ $product->get_id() ] ) ) {
				$price_html = $this->has_get_price_html_filter[ $product->get_id() ];
			} else {

				$old_price = apply_filters( 'ywdpd_change_base_price', floatval( $product->get_price() ), $product );
				$new_price = $this->get_quantity_price( $old_price, $product );
				if ( $old_price !== $new_price && isset( $this->valid_bulk_rule[ $product->get_id() ] ) ) {
					$price_html = YWDPD_Utils::get_product_new_price_html( $old_price, $new_price, $product );
					$price_html = apply_filters( 'ywdpd_get_product_price_html', $price_html, $old_price, $new_price, $product );
				}
				$this->has_get_price_html_filter[ $product->get_id() ] = $price_html;

			}
		}
		$this->add_price_filters();

		return $price_html;
	}

	/**
	 * Return the right price html with the Dynamic price format
	 *
	 * @param string              $price_html The price html.
	 * @param WC_Product_Variable $product The variable product.
	 *
	 * @return string
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_product_variable_price_html( $price_html, $product ) {

		if ( $this->can_calculate_dynamic_price( $price_html, $product ) ) {

			if ( isset( $this->has_get_price_html_filter[ $product->get_id() ] ) ) {
				$price_html = $this->has_get_price_html_filter[ $product->get_id() ];
			} else {

				$this->remove_price_filters();
				$min_price = $product->get_variation_price( 'min', true );
				$max_price = $product->get_variation_price( 'max', true );

				$min_dynamic_price = $this->get_variation_dynamic_price( $product );
				$max_dynamic_price = $this->get_variation_dynamic_price( $product, 'max' );

				$has_dynamic_price = ( $min_price !== $min_dynamic_price || $max_price !== $max_dynamic_price );

				if ( ! empty( $min_dynamic_price ) && ! empty( $max_dynamic_price ) && $has_dynamic_price ) {
					$min_dynamic_price = wc_get_price_to_display(
						$product,
						array(
							'qty'   => 1,
							'price' => $min_dynamic_price,
						)
					);
					$max_dynamic_price = wc_get_price_to_display(
						$product,
						array(
							'qty'   => 1,
							'price' => $max_dynamic_price,
						)
					);

					if ( $min_price !== $max_price ) {
						$original_price_html = apply_filters( 'ywdpd_change_variable_products_html_regular_price', wc_format_price_range( $min_price, $max_price ), $min_price, $max_price, $product );
					} else {
						$original_price_html = wc_price( $min_price );
					}

					if ( $min_dynamic_price !== $max_dynamic_price ) {
						$dynamic_price_html = apply_filters( 'ywdpd_change_variable_products_html_discount_price', wc_format_price_range( $min_dynamic_price, $max_dynamic_price ), $min_dynamic_price, $max_dynamic_price, $product );
					} else {
						$dynamic_price_html = wc_price( $min_dynamic_price );
					}

					$min_percentage_discount = YWDPD_Utils::get_discount_percentage( $min_dynamic_price, $min_price );
					$max_percentage_discount = YWDPD_Utils::get_discount_percentage( $max_dynamic_price, $max_price );
					if ( $min_percentage_discount > 0 ) {
						if ( $min_percentage_discount !== $max_percentage_discount ) {
							$percentage_discount = YWDPD_Utils::get_discount_percentage_from_to_html( $min_percentage_discount, $max_percentage_discount );
						} else {
							$percentage_discount = YWDPD_Utils::get_discount_percentage_html( $min_percentage_discount );
						}
					} else {
						$percentage_discount = '';
					}

					$price_html = YWDPD_Utils::get_formatted_price_html( $original_price_html, $dynamic_price_html, $percentage_discount, $product ) . $product->get_price_suffix();

					$price_html = apply_filters( 'ywdpd_get_variable_price_html', $price_html, $min_price, $max_price, $min_dynamic_price, $max_dynamic_price, $product );

					$this->has_get_price_html_filter[ $product->get_id() ] = $price_html;
				}
			}
			$this->add_price_filters();

		}

		return $price_html;
	}

	/**
	 * Return the dynamic price
	 *
	 * @param float      $price The base price.
	 * @param WC_Product $product The product.
	 * @param int        $qt The quantity to check.
	 * @param bool       $check_loop Force to check if is rule is enabled on loop.
	 *
	 * @return float
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_dynamic_price( $price, $product, $qt = 1, $check_loop = false ) {

		$price_rules = ywdpd_get_price_rules_by_type( array( 'bulk', 'discount_whole', 'category_discount' ) );

		if ( is_array( $price_rules ) && count( $price_rules ) > 0 ) {

			foreach ( $price_rules as $price_rule ) {
				/**
				 * Current price rule
				 *
				 * @var YWDPD_Price_Rule $price_rule
				 */

				if ( $price_rule->is_valid() ) {

					$is_valid_to_apply = $this->rule_is_valid_for_product( $product, $price_rule );
					if ( $is_valid_to_apply ) {
						$price = $price_rule->get_discounted_price( $price, $qt, $product );
						break;
					}
				}
			}
		}

		return $price;
	}

	/**
	 * Check if the rule is valid for product.
	 *
	 * @param WC_Product       $product The product.
	 * @param YWDPD_Price_Rule $price_rule The rule.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function rule_is_valid_for_product( $product, $price_rule ) {
		$is_enabled_for_on_sale_products = ! ( $product->is_on_sale() && $price_rule->is_disabled_on_sale() );

		$show_in_loop = is_single() || ( method_exists( $price_rule, 'can_show_discount_in_loop' ) && $price_rule->can_show_discount_in_loop() );
		$is_valid     = false;

		if ( apply_filters( 'ywdpd_rule_is_valid_for_product', true, $product, $price_rule ) ) {
			if ( $is_enabled_for_on_sale_products && $show_in_loop && $price_rule->is_valid() ) {
				$rule_adjustment_to_active = $price_rule->is_enabled_apply_adjustment_to();

				if ( $rule_adjustment_to_active ) {
					$is_valid = $price_rule->is_valid_to_adjust( $product, 'variable' === $product->get_type() );
				} else {
					$is_valid = $price_rule->is_valid_to_apply( $product, 'variable' === $product->get_type() );

				}
			}
		}

		return $is_valid;
	}

	/**
	 * Check if a rule has already been applied in the cart
	 *
	 * @param YWDPD_Price_Rule $price_rule The price rule.
	 *
	 * @return  bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function is_price_rule_valid_in_cart( $price_rule ) {

		$is_valid = false;

		if ( ! is_null( WC()->cart ) && ! WC()->cart->is_empty() ) {

			foreach ( WC()->cart->get_cart_contents() as $cart_item_key => $cart_item ) {
				$product = $cart_item['data'];

				if ( $price_rule->is_valid_to_apply( $product ) && ! isset( $cart_item['has_bulk_applied'] ) ) {
					$is_valid = true;
					break;
				}
			}
		}

		return $is_valid;
	}

	/**
	 * Get the min or max dynamic price for the variable product
	 *
	 * @param WC_Product_Variable $product The variable product.
	 * @param string              $min_max Min or max value.
	 * @param bool                $for_display Return price for display or not.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_variation_dynamic_price( $product, $min_max = 'min', $for_display = false ) {

		$rule          = $this->get_valid_bulk_rule_for_product( $product );
		$prices        = $product->get_variation_prices();
		$prices        = isset( $prices['price'] ) ? $prices['price'] : array();
		$min_price     = current( $prices );
		$max_price     = end( $prices );
		$current_price = '';
		if ( false !== $rule ) {
			$current_price        = 'min' === $min_max ? $min_price : $max_price;
			$current_variation_id = array_search( $current_price, $prices );

			if ( $min_price === $max_price ) {
				foreach ( $prices as $variation_id => $price ) {
					$variation = wc_get_product( $variation_id );
					$price     = apply_filters( 'ywdpd_change_base_price', $price, $variation );
					if ( $this->rule_is_valid_for_product( $variation, $rule ) ) {
						$this->valid_bulk_rule[ $variation_id ] = $rule;
						$new_price                              = $this->get_quantity_price( $price, $variation, $rule );
					} else {
						$new_price = $price;
					}
					if ( $new_price < $current_price ) {
						$current_price = $new_price;
					}
				}
			} else {

				$variation     = wc_get_product( $current_variation_id );
				$current_price = apply_filters( 'ywdpd_change_base_price', $current_price, $variation );
				if ( $this->rule_is_valid_for_product( $variation, $rule ) ) {
					$this->valid_bulk_rule[ $current_variation_id ] = $rule;
					$new_price                                      = $this->get_quantity_price( $current_price, $variation, $rule );
				} else {
					$new_price = $current_price;
				}
				if ( $new_price < $current_price ) {
					$current_price = $new_price;
				}
			}

		}

		return $current_price;
	}


	/**
	 * Return the right price for a "Bulk" rule ( quantity table,whole discount and category discount )
	 *
	 * @param float      $price The product price.
	 * @param WC_Product $product The product.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_quantity_price( $price, $product, $rule = false ) {

		if ( false === $rule ) {
			$rule = $this->get_valid_bulk_rule_for_product( $product );
		}
		if ( false !== $rule ) {
			$show_minimum_price = YITH_WC_Dynamic_Options::get_default_price();
			if ( 'bulk' === $rule->get_discount_mode() && 'max' === $show_minimum_price ) {

				$table_price_rules = $rule->get_rules();
				$max_rule_discount = end( $table_price_rules );
				$price             = $rule->get_discount_amount( $price, $max_rule_discount );
			} else {
				$price = $rule->get_discounted_price( $price, 1, $product );

			}
		}

		return $price;
	}

	/**
	 * Return a valid bulk rule if exist, otherwise false
	 *
	 * @param WC_Product $product The product.
	 *
	 * @return YWDPD_Quantity_Table|YWDPD_Discount_Whole|YWDPD_Category_Discount|bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_valid_bulk_rule_for_product( $product ) {
		$product_id = $product->get_id();
		$rule       = false;
		$this->remove_price_filters();

		if ( isset( $this->valid_bulk_rule[ $product_id ] ) ) {
			$rule = $this->valid_bulk_rule[ $product_id ];
		} else {

			$this->all_bulk_rules = ywdpd_get_price_rules_by_type( array( 'bulk', 'discount_whole', 'category_discount' ) );

			if ( is_array( $this->all_bulk_rules ) && count( $this->all_bulk_rules ) > 0 ) {

				$found = false;

				foreach ( $this->all_bulk_rules as $price_rule ) {
					/**
					 * Current price rule
					 *
					 * @var YWDPD_Price_Rule $price_rule
					 */

					$is_valid = $this->rule_is_valid_for_product( $product, $price_rule );

					if ( $is_valid ) {
						$this->valid_bulk_rule[ $product_id ] = $price_rule;
						$rule                                 = $this->valid_bulk_rule[ $product_id ];
						$found                                = true;
					}

					if ( $found ) {
						break;
					}

				}
			}
		}
		$this->add_price_filters();

		return $rule;
	}


	/**
	 * Init the hook to show quantity table and notices in the product page
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function init_tables_and_notices() {
		$show_tables = YITH_WC_Dynamic_Options::show_quantity_table();

		if ( yith_plugin_fw_is_true( $show_tables ) ) {
			$this->init_quantity_tables();
		}

		$show_notice = YITH_WC_Dynamic_Options::show_note_on_products();

		if ( yith_plugin_fw_is_true( $show_notice ) ) {
			$this->init_product_notices();
		}
	}

	/**
	 * Show the table in the right position
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function init_quantity_tables() {
		$table_position = YITH_WC_Dynamic_Options::get_quantity_table_position();

		if ( 'shortcode' !== $table_position ) {
			$priority_single_add_to_cart = has_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart' );
			$priority_single_excerpt     = has_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_excerpt' );

			$custom_hook = apply_filters( 'ywdpd_table_custom_hook', array() );
			if ( ! empty( $custom_hook ) && isset( $custom_hook['hook'] ) ) {
				$hook     = $custom_hook['hook'];
				$priority = isset( $custom_hook['priority'] ) ? $custom_hook['priority'] : 10;
				add_action( $hook, array( $this, 'show_table_quantity' ), $priority );
			} else {
				switch ( $table_position ) {
					case 'after_add_to_cart':
						if ( $priority_single_add_to_cart ) {
							$priority_single_add_to_cart ++;
						} else {
							$priority_single_add_to_cart = 32;
						}

						add_action( 'woocommerce_single_product_summary', array( $this, 'show_table_quantity' ), $priority_single_add_to_cart );
						break;
					case 'before_excerpt':
						if ( $priority_single_excerpt ) {
							$priority_single_excerpt --;
						} else {
							$priority_single_excerpt = 18;
						}
						add_action( 'woocommerce_single_product_summary', array( $this, 'show_table_quantity' ), $priority_single_excerpt );
						break;
					case 'after_excerpt':
						if ( $priority_single_excerpt ) {
							$priority_single_excerpt ++;
						} else {
							$priority_single_excerpt = 22;
						}
						add_action( 'woocommerce_single_product_summary', array( $this, 'show_table_quantity' ), $priority_single_excerpt );

						break;
					case 'after_meta':
						$priority_after_meta = has_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_meta' );
						if ( $priority_after_meta ) {
							$priority_after_meta ++;

						} else {
							$priority_after_meta = 42;
						}
						add_action( 'woocommerce_single_product_summary', array( $this, 'show_table_quantity' ), $priority_after_meta );
						break;
					default:
						if ( $priority_single_add_to_cart ) {

							$priority_single_add_to_cart --;

						} else {
							$priority_single_add_to_cart = 28;
						}
						add_action( 'woocommerce_single_product_summary', array( $this, 'show_table_quantity' ), $priority_single_add_to_cart );
						break;
				}
			}
		}
	}

	/**
	 * Show the rule notices in the right position
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function init_product_notices() {
		$position                    = YITH_WC_Dynamic_Options::get_show_note_on_products_place();
		$priority_single_add_to_cart = has_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_add_to_cart' );
		$priority_single_excerpt     = has_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_excerpt' );

		if ( 'shortcode' !== $position ) {
			$custom_hook = apply_filters( 'ywdpd_note_custom_hook', array() );

			if ( ! empty( $custom_hook ) && isset( $custom_hook['hook'] ) ) {
				$hook     = $custom_hook['hook'];
				$priority = isset( $custom_hook['priority'] ) ? $custom_hook['priority'] : 10;
				add_action( $hook, array( $this, 'show_note_on_products' ), $priority );

				return;
			}

			switch ( $position ) {
				case 'after_add_to_cart':
					if ( $priority_single_add_to_cart ) {
						$priority_single_add_to_cart ++;
					} else {
						$priority_single_add_to_cart = 32;
					}
					add_action( 'woocommerce_single_product_summary', array( $this, 'show_note_on_products' ), $priority_single_add_to_cart );
					break;
				case 'before_excerpt':
					if ( $priority_single_excerpt ) {
						$priority_single_excerpt --;
					} else {
						$priority_single_excerpt = 18;
					}
					add_action( 'woocommerce_single_product_summary', array( $this, 'show_note_on_products' ), $priority_single_excerpt );
					break;
				case 'after_excerpt':
					if ( $priority_single_excerpt ) {
						$priority_single_excerpt ++;
					} else {
						$priority_single_excerpt = 22;
					}
					add_action( 'woocommerce_single_product_summary', array( $this, 'show_note_on_products' ), $priority_single_excerpt );

					break;
				case 'after_meta':
					$priority_after_meta = has_action( 'woocommerce_single_product_summary', 'woocommerce_template_single_meta' );
					if ( $priority_after_meta ) {
						$priority_after_meta ++;
					} else {
						$priority_after_meta = 42;
					}
					add_action( 'woocommerce_single_product_summary', array( $this, 'show_note_on_products' ), $priority_after_meta );
					break;
				default:
					if ( $priority_single_add_to_cart ) {
						$priority_single_add_to_cart --;
					} else {
						$priority_single_add_to_cart = 28;
					}
					add_action( 'woocommerce_single_product_summary', array( $this, 'show_note_on_products' ), $priority_single_add_to_cart );
					break;
			}
		}
	}

	/**
	 * Show the quantity table
	 *
	 * @param false|WC_Product $product The product object.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function show_table_quantity( $product = false ) {

		if ( ! $product || is_array( $product ) ) {
			global $product;
		}
		// @codingStandardsIgnoreStart
		if ( function_exists( 'apply_shortcodes' ) ) {
			echo apply_shortcodes( '[yith_ywdpd_quantity_table product=' . $product->get_id() . ']' );
		} else {
			echo do_shortcode( '[yith_ywdpd_quantity_table product=' . $product->get_id() . ']' );
		}
		// @codingStandardsIgnoreEnd
	}

	/**
	 * Show the notices
	 *
	 * @param false|WC_Product $product The product.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function show_note_on_products( $product = false ) {

		if ( ! $product || is_array( $product ) ) {
			global $product;
		}
		// @codingStandardsIgnoreStart
		if ( function_exists( 'apply_shortcodes' ) ) {
			echo apply_shortcodes( '[yith_ywdpd_product_note product=' . $product->get_id() . ']' );
		} else {
			echo do_shortcode( '[yith_ywdpd_product_note product=' . $product->get_id() . ']' );
		}
		// @codingStandardsIgnoreEnd
	}

	/**
	 * Add if exist the quantity table in the variation data
	 *
	 * @param array                $variation_data The variation data.
	 * @param WC_Product_Variable  $variable The variable product.
	 * @param WC_Product_Variation $variation The variation product.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function add_quantity_table_to_available_variation( $variation_data, $variable, $variation ) {
		ob_start();
		$this->show_table_quantity( $variation );
		$variation_data['table_price'] = ob_get_clean();

		return $variation_data;
	}

	/**
	 * Force if show the variation price or not.
	 *
	 * @param bool                 $show Show or not the variation price.
	 * @param WC_Product_Variable  $variable The variable product.
	 * @param WC_Product_Variation $variation The variation product.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function show_variation_price( $show, $variable, $variation ) {

		if ( ! $show ) {
			$show = isset( $this->valid_bulk_rule[ $variation->get_id() ] );
		}

		return $show;
	}

	/**
	 * Remove the price filters
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function remove_price_filters() {
		remove_filter( 'woocommerce_get_price_html', array( $this, 'get_product_price_html' ), 10 );
		remove_filter( 'woocommerce_variable_price_html', array( $this, 'get_product_variable_price_html' ), 10 );
	}

	/**
	 * Add the price filters
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function add_price_filters() {

		add_filter( 'woocommerce_get_price_html', array( $this, 'get_product_price_html' ), 10, 2 );
		add_filter( 'woocommerce_variable_price_html', array( $this, 'get_product_variable_price_html' ), 10, 2 );

	}

	/**
	 * Show the new price if the cart item has a dynamic rule
	 *
	 * @param string $price_html The old cart item price.
	 * @param array  $cart_item The cart item object.
	 * @param string $cart_item_key The cart item key.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function replace_cart_item_price( $price_html, $cart_item, $cart_item_key ) {
		do_action( 'ywdpd_before_replace_cart_item_price', $price_html, $cart_item, $cart_item_key );

		if ( isset( $cart_item['ywdpd_discounts'] ) ) {
			$original_price   = apply_filters( 'ywdpd_cart_item_display_price', floatval( $cart_item['ywdpd_discounts']['display_price'] ), $cart_item );
			$discounted_price = apply_filters( 'ywdpd_cart_item_adjusted_price', floatval( $cart_item['ywdpd_discounts']['price_adjusted'] ), $cart_item );
			$tax_mode         = is_callable( array( WC()->cart, 'get_tax_price_display_mode' ) ) ? WC()->cart->get_tax_price_display_mode() : WC()->cart->tax_display_cart;

			if ( 'excl' === $tax_mode ) {
				$new_price = floatval( wc_get_price_excluding_tax( $cart_item['data'], array( 'price' => $discounted_price ) ) );
			} else {
				$new_price = floatval( wc_get_price_including_tax( $cart_item['data'], array( 'price' => $discounted_price ) ) );
			}

			$how_show              = YITH_WC_Dynamic_Options::how_show_special_offer_subtotal();
			$can_show              = ! isset( $cart_item['has_special_offer'] ) || ( isset( $cart_item['has_special_offer'] ) && 'unit_price' === $how_show ) || defined( 'DOING_AJAX' );
			$show_discounted_price = apply_filters( 'ywdpd_show_discount_cart_item_price', $original_price !== $new_price, $original_price, $discounted_price, $cart_item );
			if ( $can_show && $show_discounted_price ) {
				$price_html = wc_format_sale_price( wc_price( $original_price ), wc_price( $new_price ) );
			} else {
				$price_html = wc_price( $new_price );
			}

			$price_html = apply_filters( 'ywdpd_replace_cart_item_price', $price_html, $new_price, $cart_item, $cart_item_key );
		}
		do_action( 'ywdpd_after_replace_cart_item_price', $price_html, $cart_item, $cart_item_key );

		return $price_html;
	}

	/**
	 * Replace cart item subtotal if a special offer is applied
	 *
	 * @param string $subtotal , The old subtotal html.
	 * @param array  $cart_item , Tha cart item array.
	 * @param string $cart_item_key , The cart item key.
	 *
	 * @return string;
	 * @author YITH
	 * @since 3.0
	 */
	public function replace_cart_item_subtotal( $subtotal, $cart_item, $cart_item_key ) {
		do_action( 'ywdpd_before_replace_cart_item_subtotal', $subtotal, $cart_item, $cart_item_key );

		if ( isset( $cart_item['ywdpd_discounts'] ) ) {
			$how_show       = YITH_WC_Dynamic_Options::how_show_special_offer_subtotal();
			$original_price = apply_filters( 'ywdpd_cart_item_display_price', floatval( $cart_item['ywdpd_discounts']['display_price'] ), $cart_item );
			$new_price      = floatval( $cart_item['data']->get_price( 'edit' ) );

			if ( 'subtotal' === $how_show && isset( $cart_item['ywdpd_discounts'] ) && isset( $cart_item['has_special_offer'] ) ) {
				if ( $new_price !== $original_price ) {
					$old_product = wc_get_product( $cart_item['data'] );
					$old_product->set_price( $original_price );
					$old_product_subtotal = wc_price( $original_price * $cart_item['quantity'] );
					$subtotal             = wc_format_sale_price( $old_product_subtotal, $subtotal );
					$special_offer_name   = '';
					if ( isset( $cart_item['ywdpd_discounts']['applied_discounts'] ) && count( $cart_item['ywdpd_discounts']['applied_discounts'] ) > 0 ) {
						$rules_to_check = array( 'bogo', 'special_offer', 'gift_products' );
						foreach ( $cart_item['ywdpd_discounts']['applied_discounts'] as $applied_discount ) {
							if ( in_array( $applied_discount['discount_mode'], $rules_to_check, true ) ) {

								$rule_id            = $applied_discount['set_id'];
								$rule               = ywdpd_get_rule( $rule_id );
								$rule_name          = $rule instanceof YWDPD_Price_Rule ? $rule->get_name() : '';
								$special_offer_name = apply_filters( 'ywdpd_special_offer_name_subtotal', $rule_name );
								break;
							}
						}
					}
					$subtotal = sprintf( "<div class='ywdpd_subtotal_row'>%s<p><small><strong>%s</strong></small></p></div>", $subtotal, $special_offer_name );
				}
			} elseif ( isset( $cart_item['has_bogo_applied'] ) && floatval( 0 ) === $new_price ) {
				$old_product = wc_get_product( $cart_item['data'] );
				$old_product->set_price( $original_price );
				$old_product_subtotal = wc_price( $original_price * $cart_item['quantity'] );
				$subtotal             = wc_format_sale_price( $old_product_subtotal, $subtotal );
				$rule                 = ywdpd_get_rule( $cart_item['rule_id'] );
				$rule_name            = $rule instanceof YWDPD_Price_Rule ? $rule->get_name() : '';
				$subtotal             = sprintf( "<div class='ywdpd_subtotal_row'>%s<p><small><strong>%s</strong></small></p></div>", $subtotal, $rule_name );

			}
		} elseif ( isset( $cart_item['ywdpd_is_gift_product'] ) ) {
			$rule      = ywdpd_get_rule( $cart_item['ywdpd_rule_id'] );
			$rule_name = $rule instanceof YWDPD_Price_Rule ? $rule->get_name() : '';
			$subtotal  = sprintf( "<div class='ywdpd_subtotal_row'>%s<p><small><strong>%s</strong></small></p></div>", $subtotal, $rule_name );
		}
		do_action( 'ywdpd_after_replace_cart_item_subtotal', $subtotal, $cart_item, $cart_item_key );

		return $subtotal;
	}

	/**
	 * Store in the order item the dynamic rules applied
	 *
	 * @param WC_Order_Item_Product $item_product The  product item object.
	 * @param string                $cart_key The cart item key.
	 * @param array                 $cart_item The cart item.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function save_dynamic_rules_in_order_item( $item_product, $cart_key, $cart_item ) {

		if ( isset( $cart_item['ywdpd_discounts'] ) ) {
			$applied_discount = $cart_item['ywdpd_discounts'];
			$item_product->add_meta_data( '_ywdpd_discounts', $applied_discount );
			$item_product->save();
		} elseif ( isset( $cart_item['ywdpd_is_gift_product'] ) ) {
			$rule     = ywdpd_get_rule( $cart_item['ywdpd_rule_id'] );
			$tax_mode = is_callable(
				array(
					WC()->cart,
					'get_tax_price_display_mode',
				)
			) ? WC()->cart->get_tax_price_display_mode() : WC()->cart->tax_display_cart;

			$display_price = 'excl' === $tax_mode ? wc_get_price_excluding_tax( $cart_item['data'] ) : wc_get_price_including_tax( $cart_item['data'] );
			$discount_data = array(
				'by'                => array(
					$rule->get_discount_mode(),
				),
				'set_id'            => $rule->get_id(),
				'price_base'        => $cart_item['data']->get_price(),
				'display_price'     => $display_price,
				'price_adjusted'    => 0,
				'applied_discounts' => array(
					array(
						'by'             => $rule->get_discount_mode(),
						'set_id'         => $rule->get_id(),
						'price_base'     => $cart_item['data']->get_price(),
						'price_adjusted' => 0,
						'discount_mode'  => $rule->get_discount_mode(),
					),
				),
			);
			$item_product->add_meta_data( '_ywdpd_discounts', $discount_data );
			$item_product->save();
		}
	}

	/**
	 * Save dynamic coupon information
	 *
	 * @param WC_Order_Item_Coupon $item_coupon The coupon item object.
	 * @param string               $code The coupon code.
	 *
	 * @author YITH
	 * @since 3.4.0
	 */
	public function save_dynamic_coupon_name( $item_coupon, $code ) {
		$dynamic_code = ywdpd_dynamic_pricing_discounts()->get_cart_rules_manager()->get_coupon_code();

		if ( $dynamic_code === $code ) {
			$dynamic_coupon_name = ywdpd_dynamic_pricing_discounts()->get_cart_rules_manager()->get_coupon_code_details();
			$item_coupon->add_meta_data( '_ywdpd_coupon_info', $dynamic_coupon_name );
			$item_coupon->save();
		}

	}

	/**
	 * Show the custom meta for the item.
	 *
	 * @param int           $item_id The item id.
	 * @param WC_Order_Item $item The order item object.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function format_custom_meta_data( $item_id, $item ) {
		if ( YITH_WC_Dynamic_Options::can_show_rule_details() ) {
			$dynamic_rules = $item->get_meta( '_ywdpd_discounts' );
			if ( ! empty( $dynamic_rules ) ) {
				$custom_meta = '';

				foreach ( $dynamic_rules['applied_discounts'] as $applied_discount ) {

					if ( isset( $applied_discount['set_id'] ) ) {
						$rule_id = $applied_discount['set_id'];
						$rule    = ywdpd_get_rule( $rule_id );
					} else {
						$rule = $applied_discount['by'];
					}
					if ( $rule instanceof YWDPD_Price_Rule && ! empty( $rule->get_name() ) ) {
						$custom_meta .= '<li>' . apply_filters( 'ywdpd_rule_name', $rule->get_name(), $rule, $applied_discount ) . '</li>';
					}
				}
				if ( ! empty( $custom_meta ) ) { ?>
					<ul class="wc-item-meta">
						<span style="font-weight: bold;"><?php esc_html_e( 'Offer applied:', 'ywdpd' ); ?></span>
						<?php echo wp_kses_post( $custom_meta ); ?>
					</ul>
					<?php
				}
			}
		}
	}

	/**
	 * Show the discount amount info in cart
	 *
	 * @author YITH
	 * @since 2.0.0
	 */
	public function show_total_discount_message_on_cart() {
		$message          = '';
		$original_message = YITH_WC_Dynamic_Options::get_discount_info_message();

		$coupons_applied = WC()->cart->get_applied_coupons();
		$tax_excluded    = 'tax_excluded' === YITH_WC_Dynamic_Options::how_calculate_discounts();
		$amount          = 0;

		foreach ( $coupons_applied as $coupon_code ) {

			$coupon = new WC_Coupon( $coupon_code );
			$meta   = $coupon->get_meta( 'ywdpd_coupon' );
			if ( ! empty( $meta ) ) {
				$amount = WC()->cart->get_coupon_discount_amount( $coupon->get_code(), $tax_excluded );
				break;
			}
		}

		$subtotal = WC()->cart->get_subtotal();
		if ( ! $tax_excluded ) {
			$subtotal += WC()->cart->get_subtotal_tax();
		}

		if ( $amount > 0 ) {

			$perc_discount  = round( $amount / $subtotal, 2 ) * 100;
			$price_discount = $subtotal - $amount;
			$message        = str_replace( '%total_discount_percentage%', $perc_discount . '%', $original_message );
			$message        = str_replace( '%total_discount_price%', wc_price( $price_discount ), $message );
			$message        = sprintf( '<div class="ywdpd_single_cart_notice">%s</div>', $message );
		}

		if ( ! empty( $message ) ) {
			?>
			<tr class="dynamic-discount">
				<td colspan="2" data-title="<?php _e( 'Discount', 'ywdpd' ); ?>"><?php echo $message; ?></td>
			</tr>
			<?php
		}
	}

	/**
	 * Show the cart notices
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function show_cart_notices() {
		// @codingStandardsIgnoreStart
		if ( function_exists( 'apply_shortcodes' ) ) {
			echo apply_shortcodes( '[yith_ywdpd_cart_notice]' );
		} else {
			echo do_shortcode( '[yith_ywdpd_cart_notice]' );
		}
		// @codingStandardsIgnoreEnd
	}
}
