<?php
/**
 * Gift products class
 *
 * @package YITH WooCommerce Dynamic Pricing and Discounts Premium
 * @since   1.0.0
 * @version 1.6.0
 * @author  YITH
 */

if ( ! defined( 'ABSPATH' ) || ! defined( 'YITH_YWDPD_VERSION' ) ) {
	exit; // Exit if accessed directly.
}

if ( ! class_exists( 'YITH_WC_Dynamic_Pricing_Gift_Product' ) ) {

	/**
	 * Class YITH_WC_Dynamic_Pricing_Gift_Product
	 */
	class YITH_WC_Dynamic_Pricing_Gift_Product {

		/**
		 * Instance.
		 *
		 * @var YITH_WC_Dynamic_Pricing_Gift_Product
		 */
		protected static $instance;

		/**
		 * List of gift rules.
		 *
		 * @var array
		 */
		private $gift_rules = array();

		/**
		 * List of rules to apply.
		 *
		 * @var array
		 */
		private $gift_rules_to_apply = array();

		/**
		 * Constructor
		 */
		private function __construct() {

			add_action( 'init', array( $this, 'load_gift_rules' ), 20 );
			add_action( 'woocommerce_add_to_cart', array( $this, 'check_if_apply_rules' ), 25 );
			add_action( 'woocommerce_cart_loaded_from_session', array( $this, 'check_if_apply_rules' ), 110 );
			add_filter( 'woocommerce_after_calculate_totals', array( $this, 'update_gift_products' ), 20, 1 );
			add_filter( 'woocommerce_update_cart_validation', array( $this, 'check_quantity_security' ), 25, 2 );

			add_filter( 'woocommerce_add_cart_item', array( $this, 'change_price_gift_product' ), 20, 2 );
			add_filter( 'woocommerce_get_cart_item_from_session', array( $this, 'change_price_gift_product' ), 30, 2 );

			add_action( 'wp_footer', array( $this, 'show_popup' ), 25 );
		}

		/**
		 * Prevent the clone of object
		 *
		 * @author YITH
		 * @since 3.0.0
		 */
		private function __clone() {
		}

		/**
		 * Prevent unserialize the object.
		 *
		 * @author YITH
		 * @since 3.0.0
		 */
		public function __wakeup() {
		}

		/**
		 * Returns single instance of the class
		 *
		 * @return YITH_WC_Dynamic_Pricing_Gift_Product
		 * @since 1.0.0
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Load the gift rules
		 */
		public function load_gift_rules() {
			$this->gift_rules = ywdpd_get_price_rules_by_type( 'gift_products' );

		}

		/**
		 * Check if the rule can be applied.
		 */
		public function check_if_apply_rules() {

			if ( ! is_null( WC()->cart ) && ! WC()->cart->is_empty() ) {

				$no_gift_products = $this->get_cart_products();
				$add_rule         = true;

				if ( count( $no_gift_products ) > 0 ) {
					foreach ( $no_gift_products as $cart_item_key => $cart_item ) {
						foreach ( $this->gift_rules as $rule ) {
							/**
							 * The gift rules
							 *
							 * @var YWDPD_Gift_Products $rule
							 */
							$disabled_with_other_rule = ! $rule->is_possible_apply_other_rules();

							if ( $rule->is_valid() && $rule->is_valid_for_cart( $cart_item['data'] ) ) {
								if ( $add_rule ) {
									$this->gift_rules_to_apply[ $rule->get_id() ] = $rule;
								}

								if ( $disabled_with_other_rule ) {
									$add_rule = false;
								} else {
									$add_rule = true;
								}
							}
						}
					}
				} else {
					// remove all gift products.
					$this->remove_all_gift_product();
				}
			}
		}

		/**
		 * Remove all gift product in the cart
		 */
		public function remove_all_gift_product() {

			foreach ( WC()->cart->cart_contents as $cart_item_key => $cart_item ) {

				$is_gift_product = isset( $cart_item['ywdpd_is_gift_product'] );
				if ( $is_gift_product ) {

					/**
					 * Product.
					 *
					 * @var WC_Product $product
					 */
					$product = $cart_item['data'];
					WC()->cart->remove_cart_item( $cart_item_key );
					/* translators: name of product */
					wc_add_notice( sprintf( __( 'Gift %s removed properly', 'ywdpd' ), $product->get_formatted_name() ) );
				}
			}
		}

		/**
		 * Remove gift product from cart.
		 *
		 * @param integer $rule_id Rule id.
		 */
		public function remove_gift_product( $rule_id ) {

			foreach ( WC()->cart->cart_contents as $cart_item_key => $cart_item ) {

				$is_gift_product = isset( $cart_item['ywdpd_is_gift_product'] );
				$cart_rule_id    = isset( $cart_item['ywdpd_rule_id'] ) ? $cart_item['ywdpd_rule_id'] : false;

				if ( $is_gift_product && $cart_rule_id === $rule_id ) {
					$product = $cart_item['data'];
					WC()->cart->remove_cart_item( $cart_item_key );
					/* translators: name of product */
					wc_add_notice( sprintf( __( 'Gift %s removed properly', 'ywdpd' ), $product->get_formatted_name() ) );
				}
			}
		}


		/**
		 * Get the product on cart that aren't gift.
		 *
		 * @return array
		 */
		public function get_cart_products() {

			$products_in_cart = array();
			foreach ( WC()->cart->cart_contents as $cart_item_key => $cart_item ) {

				$is_gift_product = isset( $cart_item['ywdpd_is_gift_product'] );

				if ( ! $is_gift_product ) {

					$products_in_cart[ $cart_item_key ] = $cart_item;
				}
			}

			return $products_in_cart;
		}


		/**
		 * Get the gift rules.
		 */
		public function get_gift_product_to_add_popup() {
			$gift_rules_valid = array();

			if ( count( $this->gift_rules_to_apply ) > 0 ) {

				foreach ( $this->gift_rules_to_apply as $rule ) {
					/**
					 * The gift rule
					 *
					 * @var YWDPD_Gift_Products $rule
					 */
					$key           = $rule->get_id();
					$items_in_cart = $rule->get_total_gift_product_in_cart();
					$allowed_item  = $rule->get_amount_gift_product_allowed();
					$rule_text     = ywdpd_get_note( $rule->get_text_in_modal_gift() );
					$rule_text     = ! empty( $rule_text ) ? $rule_text : __( 'You can add {{total_to_add}} product(s) for free!', 'ywdpd' );

					if ( $items_in_cart < $allowed_item ) {
						$product_to_gift          = $rule->get_gift_product_selection();
						$gift_rules_valid[ $key ] = array(
							'text'          => $rule_text,
							'items_in_cart' => $items_in_cart,
							'allowed_item'  => $allowed_item - $items_in_cart,
							'items'         => array(
								'type'     => 'product_ids',
								'item_ids' => $product_to_gift,
							),
							'type'          => 'gift_products',
							'discount'      => array(
								'type'   => 'percentage',
								'amount' => 100,
							),

						);
					}
				}
			}

			return $gift_rules_valid;
		}

		/**
		 * Get the gift rule by id
		 *
		 * @param string|int $rule_id The rule id.
		 *
		 * @return false|YWDPD_Gift_Products
		 * @author YITH
		 * @since 3.0.0
		 */
		public function get_gift_rule_to_apply_by_id( $rule_id ) {

			return isset( $this->gift_rules_to_apply[ $rule_id ] ) ? $this->gift_rules_to_apply[ $rule_id ] : false;
		}


		/**
		 * Change gift product price.
		 *
		 * @param array  $cart_item_data Cart item data.
		 * @param string $cart_item_key Cart item key.
		 *
		 * @return array
		 */
		public function change_price_gift_product( $cart_item_data, $cart_item_key ) {

			if ( isset( $cart_item_data['ywdpd_is_gift_product'] ) ) {
				$cart_item_data['data']->update_meta_data( 'has_dynamic_price', true );
				$cart_item_data['data']->set_price( 0 );
			}

			return $cart_item_data;
		}


		/**
		 * Update gift product.
		 *
		 * @param WC_Cart $updated Bool.
		 *
		 * @return WC_Cart
		 */
		public function update_gift_products( $updated ) {

			$rule_to_remove = array();
			if ( empty( $this->gift_rules_to_apply ) ) {
				$this->check_if_apply_rules();
			}
			foreach ( WC()->cart->get_cart_contents() as $cart_item_key => $cart_item ) {

				$rule_id = isset( $cart_item['ywdpd_rule_id'] ) ? $cart_item['ywdpd_rule_id'] : false;

				if ( $rule_id && ! in_array( $rule_id, $rule_to_remove ) ) { //phpcs:ignore

					$rule = isset( $this->gift_rules_to_apply[ $rule_id ] ) ? $this->gift_rules_to_apply[ $rule_id ] : false;

					if ( ! $rule || ! $this->check_valid_single_rule( $rule ) ) {
						$rule_to_remove[] = $rule_id;

						$this->remove_gift_product( $rule_id );
					}
				}
			}

			return $updated;
		}


		/**
		 * Check if a rule is valid
		 *
		 * @param YWDPD_Gift_Products $rule Rule.
		 *
		 * @return bool
		 * @author YITH
		 * @since 1.6.0
		 */
		public function check_valid_single_rule( $rule ) {

			$products_in_cart = $this->get_cart_products();

			$valid = false;
			foreach ( $products_in_cart as $cart_item_key => $cart_item ) {

				if ( $rule->is_valid_for_cart( $cart_item['data'] ) ) {
					return true;
				}
			}

			return $valid;
		}

		/**
		 * Avoid the change quantity for gift product
		 *
		 * @param string $new_qty The new quantity.
		 * @param string $cart_item_key The cart item key.
		 *
		 * @return string
		 * @author YITH
		 * @since 3.3.4
		 */
		public function check_quantity_security( $new_qty, $cart_item_key ) {

			$cart_item = WC()->cart->cart_contents[ $cart_item_key ];
			$rule_id   = isset( $cart_item['ywdpd_rule_id'] ) ? $cart_item['ywdpd_rule_id'] : false;

			if ( $rule_id ) {
				$new_qty = '';
			}

			return $new_qty;

		}

		/**
		 * Check if can show popup in the page
		 *
		 * @auhtor YITH
		 * @since 2.1
		 */
		public function show_popup() {
			if ( is_cart() ) {
				$this->print_popup_for_gift_rules();
			}
		}

		/**
		 * Show the gift rules in cart
		 *
		 * @author YITH
		 * @since 2.1
		 */
		public function print_popup_for_gift_rules() {
			$items_to_show = $this->get_gift_product_to_add_popup();

			if ( count( $items_to_show ) > 0 ) {
				wc_get_template(
					'yith_ywdpd_popup.php',
					array(
						'items_to_show' => $items_to_show,
						'popup_class'   => 'cart',
					),
					YITH_YWDPD_TEMPLATE_PATH,
					YITH_YWDPD_TEMPLATE_PATH
				);
			}
		}
	}

}

/**
 * Unique access to instance of YITH_WC_Dynamic_Pricing_Gift_Product class
 *
 * @return YITH_WC_Dynamic_Pricing_Gift_Product
 */
function YITH_WC_Dynamic_Pricing_Gift_Product() { // phpcs:ignore WordPress.NamingConventions.ValidFunctionName
	return YITH_WC_Dynamic_Pricing_Gift_Product::get_instance();
}


