<?php
/**
 * The class that manage all.
 *
 * @package YITH WooCommerce Dynamic Pricing and Discounts\Classes\
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * This class manage all hook and load files.
 */
class YITH_WC_Dynamic_Pricing_Discounts {

	/**
	 * The static instance of the class.
	 *
	 * @var YITH_WC_Dynamic_Pricing_Discounts $instance
	 */
	protected static $instance = null;

	/**
	 * The construct, add hooks and load files.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	private function __construct() {

		$this->includes();
		add_action( 'init', array( 'YWDPD_Exclusion_List_Data_Store', 'create_table' ) );
		add_action( 'init', array( $this, 'load_text_domain' ), 0 );
		add_action( 'plugins_loaded', array( $this, 'plugin_fw_loader' ), 15 );
		add_action( 'ywcrbp_premium_init', array( $this, 'disable_old_rolebased_integration' ), 99 );
		add_action( 'plugins_loaded', array( $this, 'load_custom_table' ), 20 );
		add_action( 'wp_loaded', array( $this, 'ywdpd_set_cron' ) );
		add_action( 'ywdpd_clean_cron', array( $this, 'clear_coupons' ) );
		add_action( 'woocommerce_checkout_create_order', array( $this, 'clear_ywdpd_coupon_after_create_order' ) );
	}

	/**
	 * Prevent the clone of object
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	private function __clone() {
	}

	/**
	 * Prevent unserialize the object.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function __wakeup() {
	}

	/**
	 * Return the unique instance of the class.
	 *
	 * @return YITH_WC_Dynamic_Pricing_Discounts
	 * @since 3.0.0
	 * @author YITH
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {

			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Include all files.
	 *
	 * @author YITH
	 * @since
	 */
	public function includes() {

		// Functions file.
		require_once YITH_YWDPD_INC . 'functions.yith-wc-dynamic-pricing-update.php';
		require_once YITH_YWDPD_INC . 'functions.yith-ywdpd-deprecated.php';
		require_once YITH_YWDPD_INC . 'functions.yith-wc-dynamic-pricing.php';

		// Data Stores.
		require_once YITH_YWDPD_INC . 'data-stores/class-ywdpd-exclusion-list-data-store.php';
		// Support classes.
		require_once YITH_YWDPD_INC . 'class-yith-ywdpd-counter.php';
		require_once YITH_YWDPD_INC . 'class-yith-ywdpd-taxonomy.php';
		require_once YITH_YWDPD_INC . 'class-yith-wc-dynamic-pricing-options.php';
		require_once YITH_YWDPD_INC . 'class-ywdpd-utils.php';

		// Post Type registration.
		require_once YITH_YWDPD_INC . 'class-ywdpd-post-type.php';

		// Abstract classes.
		require_once YITH_YWDPD_INC . 'abstracts/abstract-ywdpd-data.php';
		require_once YITH_YWDPD_INC . 'abstracts/abstract-ywdpd-rule.php';
		require_once YITH_YWDPD_INC . 'abstracts/abstract-ywdpd-price-rule.php';

		// Price Rule classes.
		require_once YITH_YWDPD_INC . 'dynamic-rules/price-rules/class-ywdpd-quantity-table.php';
		require_once YITH_YWDPD_INC . 'dynamic-rules/price-rules/class-ywdpd-special-offer.php';
		require_once YITH_YWDPD_INC . 'dynamic-rules/price-rules/class-ywdpd-category-discount.php';
		require_once YITH_YWDPD_INC . 'dynamic-rules/price-rules/class-ywdpd-discount-whole.php';
		require_once YITH_YWDPD_INC . 'dynamic-rules/price-rules/class-ywdpd-gift-products.php';
		require_once YITH_YWDPD_INC . 'dynamic-rules/price-rules/class-ywdpd-bogo.php';
		require_once YITH_YWDPD_INC . '/legacy/class-ywdpd-exclude-items.php';

		// Cart rule class.
		require_once YITH_YWDPD_INC . 'dynamic-rules/cart-rules/class-ywdpd-cart-rule.php';

		// Price Rules Manager.
		require_once YITH_YWDPD_INC . 'class-yith-wc-dynamic-price-rules-manager.php';
		// Cart Rules Manager.
		require_once YITH_YWDPD_INC . 'class-yith-wc-dynamic-cart-rules-manager.php';
		// Gift rules manager.
		require_once YITH_YWDPD_INC . 'class-yith-wc-dynamic-pricing-gift-product.php';

		// Exclusion manager.
		require_once YITH_YWDPD_INC . 'class-yith-wc-dynamic-exclusion-manager.php';

		// Shortcodes.
		require_once YITH_YWDPD_INC . 'class-ywdpd-shortcodes.php';

		// Assets.
		require_once YITH_YWDPD_INC . 'class-ywdpd-assets.php';
		// Ajax.
		require_once YITH_YWDPD_INC . 'class-ywdpd-ajax.php';

		if ( defined( 'ELEMENTOR_VERSION' ) ) {
			// Elementor.
			require_once YITH_YWDPD_INC . '/integrations/elementor/class.yith-wc-dynamic-elementor.php';
		}
		// Legacy classes.
		require_once YITH_YWDPD_INC . '/legacy/class-yith-wc-dynamic-discounts.php';
		require_once YITH_YWDPD_INC . '/legacy/class-yith-wc-dynamic-pricing.php';
		require_once YITH_YWDPD_INC . '/legacy/class-yith-wc-dynamic-pricing-frontend.php';
		require_once YITH_YWDPD_INC . '/legacy/class-yith-wc-dynamic-pricing-helper.php';

		require_once YITH_YWDPD_INC . 'class-ywdpd-frontend.php';
		$this->get_frontend_manager();
		$this->get_exclusion_manager();
		// Integrations.
		require_once YITH_YWDPD_INC . '/integrations/class-yith-ywdpd-integrations.php';

		if ( self::is_request( 'admin' ) ) {
			require_once YITH_YWDPD_INC . 'admin/class-yith-wc-dynamic-pricing-admin.php';
		}

		if ( self::is_request( 'frontend' ) || self::is_request( 'ajax' ) ) {

			$this->get_gift_rules_manager();
			$this->get_price_rules_manager();
			$this->get_cart_rules_manager();
		}
	}

	/**
	 * Load Localisation files.
	 *
	 * Note: the first-loaded translation file overrides any following ones if the same translation is present.
	 *
	 * Locales found in:
	 *      - WP_LANG_DIR/yith-woocommerce-dynamic-pricing-and-discounts-premium/ywdpd-LOCALE.mo
	 *      - WP_LANG_DIR/plugins/ywdpd-LOCALE.mo
	 */
	public function load_text_domain() {
		$locale = determine_locale();
		$locale = apply_filters( 'plugin_locale', $locale, 'ywdpd' );

		unload_textdomain( 'ywdpd' );
		load_textdomain( 'ywdpd', WP_LANG_DIR . '/yith-woocommerce-dynamic-pricing-and-discounts-premium/ywdpd-' . $locale . '.mo' );
		load_plugin_textdomain( 'ywdpd', false, plugin_basename( YITH_YWDPD_DIR ) . '/languages' );
	}

	/**
	 * What type of request is this?
	 *
	 * @param string $type admin, ajax, cron or frontend.
	 *
	 * @return bool
	 * @since 3.0.0
	 * @author YITH
	 */
	public static function is_request( $type ) {
		switch ( $type ) {
			case 'admin':
				return is_admin() && ! defined( 'DOING_AJAX' ) || ( defined( 'DOING_AJAX' ) && DOING_AJAX && ( ! isset( $_REQUEST['context'] ) || ( isset( $_REQUEST['context'] ) && 'frontend' !== $_REQUEST['context'] ) ) ); //phpcs:ignore
			case 'ajax':
				return defined( 'DOING_AJAX' );
			case 'frontend':
				return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
		}

		return false;
	}

	/**
	 * Load YITH Plugin Framework
	 *
	 * @author YITH
	 * @since  3.0.0
	 */
	public function plugin_fw_loader() {
		if ( ! defined( 'YIT_CORE_PLUGIN' ) ) {
			global $plugin_fw_data;
			if ( ! empty( $plugin_fw_data ) ) {
				$plugin_fw_file = array_shift( $plugin_fw_data );
				require_once $plugin_fw_file;
			}
		}
	}

	/**
	 * Disable the old integration with YITH Role Based Prices
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function disable_old_rolebased_integration() {
		add_filter( 'ywcrbp_init_dynamic_integration', '__return_false', 9999 );
	}

	/**
	 * Instance and return the gift product manager class
	 *
	 * @return YITH_WC_Dynamic_Pricing_Gift_Product
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_gift_rules_manager() {
		return YITH_WC_Dynamic_Pricing_Gift_Product::get_instance();
	}

	/**
	 * Instance and return the price rules manager
	 *
	 * @return YWDPD_Price_Rules_Manager
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_price_rules_manager() {
		return YWDPD_Price_Rules_Manager::get_instance();
	}

	/**
	 * Instance and return the cart rules manager
	 *
	 * @return YWDPD_Cart_Rules_Manager
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_cart_rules_manager() {
		return YWDPD_Cart_Rules_Manager::get_instance();
	}

	/**
	 * Instance and return the frontend manager
	 *
	 * @return YWDPD_Frontend
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_frontend_manager() {
		return YWDPD_Frontend::get_instance();
	}

	/**
	 * Instance and return the exclusion manager
	 *
	 * @return YITH_WC_Dynamic_Exclusion_Manager
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_exclusion_manager() {
		return YITH_WC_Dynamic_Exclusion_Manager::get_instance();
	}

	/**
	 * Load the post table
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function load_custom_table() {
		require_once YITH_YWDPD_INC . 'admin/class-ywdpd-rules-table.php';
	}

	/**
	 * Delete coupon after use it
	 *
	 * @param WC_Order $order The order.
	 *
	 * @throws Exception The exception.
	 * @author YITH
	 */
	public function clear_ywdpd_coupon_after_create_order( $order ) {
		if ( version_compare( WC()->version, '3.7.0', '<' ) ) {
			$coupon_used = $order->get_used_coupons();
		} else {
			$coupon_used = $order->get_coupon_codes();
		}
		if ( $coupon_used ) {
			foreach ( $coupon_used as $coupons_code ) {
				$coupon   = new WC_Coupon( $coupons_code );
				$valid    = ywdpd_coupon_is_valid( $coupon, $order );
				$is_ywdpd = $coupon->get_meta( 'ywdpd_coupon', true );
				if ( $is_ywdpd && $valid ) {
					$coupon->delete();
				}
			}
		}
	}

	/**
	 * Clear coupons after use
	 *
	 * @since  1.0.0
	 * @author YITH
	 */
	public function clear_coupons() {

		$args = array(
			'post_type'      => 'shop_coupon',
			'posts_per_page' => - 1,
			'meta_key'       => 'ywdpd_coupon', // phpcs:ignore WordPress.DB.SlowDBQuery
			'meta_value'     => 1, // phpcs:ignore WordPress.DB.SlowDBQuery
			'date_query'     => array(
				array(
					'column' => 'post_date_gmt',
					'before' => '1 day ago',
				),
			),
		);

		$coupons = get_posts( $args );

		if ( ! empty( $coupons ) ) {
			foreach ( $coupons as $coupon ) {
				wp_delete_post( $coupon->ID, true );
			}
		}
	}

	/**
	 * Set the cron to clear coupons
	 *
	 * @author YITH
	 * @since 1.0.0
	 */
	public function ywdpd_set_cron() {
		if ( ! wp_next_scheduled( 'ywdpd_clean_cron' ) ) {
			$duration = apply_filters( 'ywdpd_set_cron_time', 'daily' );
			wp_schedule_event( time(), $duration, 'ywdpd_clean_cron' );
		}
	}
}

if ( ! function_exists( 'ywdpd_dynamic_pricing_discounts' ) ) {
	/**
	 * Return the unique instance of the class
	 */
	function ywdpd_dynamic_pricing_discounts() {
		return YITH_WC_Dynamic_Pricing_Discounts::get_instance();
	}
}
