<?php // phpcs:ignore WordPress.Files.FileName
/**
 * The class that manage the price rules
 *
 * @package  YITH\Dynamic\PricingAndDiscounts\Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The class that manage the price rules
 */
class YWDPD_Price_Rules_Manager {

	/**
	 * The unique access of the class.
	 *
	 * @var YWDPD_Price_Rules_Manager
	 */
	protected static $instance;

	/**
	 * The array that contain all price rules object.
	 *
	 * @var array
	 */
	protected $price_rules;

	/**
	 * Check if the cart
	 *
	 * @var array
	 */
	protected $cart_processed = false;

	/**
	 * All rules in the cart by cart item key.
	 *
	 * @var array
	 */
	protected $applied_rules_by_item_key = array();

	/**
	 * The construct of the class
	 */
	public function __construct() {

		add_action( 'init', array( $this, 'load_price_rules' ), 15 );
		add_action( 'woocommerce_before_calculate_totals', array( $this, 'calculate_discounts' ), 102 );
		add_action( 'woocommerce_cart_loaded_from_session', array( $this, 'calculate_discounts' ), 102 );
		add_action( 'woocommerce_add_to_cart', array( $this, 'print_popup_for_special_offer' ), 200, 6 );
		add_action( 'woocommerce_add_to_cart', array( $this, 'add_bogo_product' ), 210, 6 );
		add_action( 'woocommerce_after_calculate_totals', array( $this, 'remove_invalid_bogo' ), 210, 6 );
		add_action( 'woocommerce_cart_loaded_from_session', array( $this, 'remove_invalid_bogo' ), 210 );
		add_filter( 'woocommerce_cart_item_quantity', array( $this, 'hide_qty_field' ), 30, 3 );
		add_filter( 'woocommerce_cart_item_remove_link', array( $this, 'hide_remove_link' ), 30, 2 );
		add_action( 'woocommerce_after_cart_item_quantity_update', array( $this, 'update_cart_item_with_bogo_quantity' ), 10, 2 );
		add_action( 'woocommerce_cart_item_removed', array( $this, 'remove_cart_item_with_bogo' ), 10, 1 );
		add_filter( 'ywdpd_add_cart_item_in_clone', array( $this, 'exclude_item_from_cart' ), 110, 2 );
		add_filter( 'ywdpd_init_cart_item_counter', array( $this, 'exclude_item_from_cart' ), 110, 2 );
	}

	/**
	 * Return the unique instance of the class.
	 *
	 * @return YWDPD_Price_Rules_Manager
	 * @since 3.0.0
	 * @author YITH
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {

			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Load all prices rules
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function load_price_rules() {

		// Filter only active rule and sort it by priority desc.
		$this->price_rules = ywdpd_get_price_rules_by_type( array( 'bulk', 'special_offer', 'discount_whole', 'category_discount', 'bogo' ) );

	}

	/**
	 * Return the price rules.
	 *
	 * @return array
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_price_rule() {
		return $this->price_rules;
	}

	/**
	 * Apply the price rules in the cart.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function calculate_discounts() {
		// Get a copy of the cart object and sort it by price .
		$cloned_cart = ywdpd_clone_cart();

		uasort( $cloned_cart, 'ywdpd_sort_cart_by_price' );
		$this->set_total_target( $cloned_cart );
		$this->applied_rules_by_item_key = array();
		do_action( 'ywdpd_before_calculate_discounts' );
		if ( count( $cloned_cart ) > 0 && count( $this->price_rules ) > 0 ) {
			$cart_has_coupon = WC()->cart->has_discount();

			foreach ( $cloned_cart as $cart_item_key => $cart_item ) {
				$product    = $cart_item['data'];
				$is_on_sale = $product->is_on_sale();

				/**
				 * The current rule.
				 *
				 * @var YWDPD_Price_Rule $price_rule
				 */
				foreach ( $this->price_rules as $price_rule ) {
					$price_rule_applied = false;
					if ( $price_rule->is_valid() ) {

						// Detect if this rule could be applied for this product.
						$is_valid_to_apply = $price_rule->is_valid_to_apply( $product );

						if ( $is_valid_to_apply ) {

							$is_enabled_for_on_sale_products = ! ( $is_on_sale && $price_rule->is_disabled_on_sale() );
							$is_enabled_with_other_coupon    = ! ( $cart_has_coupon && $price_rule->is_disabled_with_other_coupon() );
							if ( $is_enabled_for_on_sale_products && $is_enabled_with_other_coupon ) {

								$is_active_adjust = $price_rule->is_enabled_apply_adjustment_to();
								if ( ! $is_active_adjust ) {
									$price_rule_applied = $price_rule->apply_rule_in_cart( $cart_item_key, $cart_item_key );
									$price_rule_applied && $this->save_the_rule_id_in_cart_key( $cart_item_key, $price_rule->get_id() );
								}
								// Add the discount in the right cart item.
								foreach ( $cloned_cart as $cart_adjustment_item_key => $cart_adjustment_item ) {
									$is_valid_to_adjust = $price_rule->is_valid_to_adjust( $cart_adjustment_item['data'] );
									// Apply the rule if is on same product OR is valid for adjustment to.
									if ( $is_valid_to_apply && $is_valid_to_adjust ) {
										$price_rule_applied = $price_rule->apply_rule_in_cart( $cart_item_key, $cart_adjustment_item_key );
										$price_rule_applied && $this->save_the_rule_id_in_cart_key( $cart_adjustment_item_key, $price_rule->get_id() );
									}
								}
							}
							// If this rule disable the others break the rule cycle.
							if ( $price_rule_applied && ! $price_rule->is_possible_apply_other_rules() ) {
								break;
							}
						}
					}
				}
			}
		}
		do_action( 'ywdpd_after_calculate_discounts' );
	}

	/**
	 * Store the applied rules in the cart
	 *
	 * @param string $cart_item_key The cart item key.
	 * @param int    $rule_id The rule id.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	protected function save_the_rule_id_in_cart_key( $cart_item_key, $rule_id ) {

		if ( ! isset( $this->applied_rules_by_item_key[ $cart_item_key ] ) ) {
			$this->applied_rules_by_item_key[ $cart_item_key ] = array( $rule_id );
		} elseif ( ! in_array( $rule_id, $this->applied_rules_by_item_key[ $cart_item_key ], true ) ) {
			$this->applied_rules_by_item_key[ $cart_item_key ][] = $rule_id;
		}
	}

	/**
	 * Show the special offer popup
	 *
	 * @param string $cart_item_key The cart key.
	 * @param int    $product_id The product id.
	 * @param int    $quantity The quantity.
	 * @param int    $variation_id The variation id.
	 * @param array  $variation The variation.
	 * @param array  $cart_item_data The data.
	 *
	 * @author YITH
	 * @since 2.0.0
	 */
	public function print_popup_for_special_offer( $cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data ) {

		if ( isset( $_REQUEST['add-to-cart'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$cart_item = WC()->cart->get_cart_item( $cart_item_key );

			if ( count( $cart_item ) > 0 ) {
				$items_to_show = $this->get_valid_special_offer_to_apply( $cart_item, $cart_item_key );
				if ( count( $items_to_show ) > 0 ) {
					wc_get_template(
						'yith_ywdpd_popup.php',
						array(
							'items_to_show' => $items_to_show,
							'popup_class'   => 'product',
						),
						YITH_YWDPD_TEMPLATE_PATH,
						YITH_YWDPD_TEMPLATE_PATH
					);
				}
			}
		}

		if ( isset( $_REQUEST['wc-ajax'] ) && 'add_to_cart' === sanitize_text_field( wp_unslash( $_REQUEST['wc-ajax'] ) ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Recommended

			set_transient( 'ywdpd_latest_cart_item_key', $cart_item_key );
		}
	}

	/**
	 * Print the popup after an ajax add to cart action
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function print_special_offer_popup_on_shop() {

		$latest_item = get_transient( 'ywdpd_latest_cart_item_key' );

		if ( false !== $latest_item && ! is_null( WC()->cart ) ) {

			$cart_item = WC()->cart->get_cart_item( $latest_item );
			if ( ! is_null( $cart_item ) ) {
				$items_to_show = $this->get_valid_special_offer_to_apply( $cart_item, $latest_item );
				if ( count( $items_to_show ) > 0 ) {
					wc_get_template(
						'yith_ywdpd_popup.php',
						array(
							'items_to_show' => $items_to_show,
							'popup_class'   => 'product',
						),
						'',
						YITH_YWDPD_TEMPLATE_PATH
					);
				}
			}
		}
		delete_transient( 'ywdpd_latest_cart_item_key' );
	}

	/**
	 * Add automatically the BOGO product in the cart .
	 *
	 * @param string $cart_item_key The cart item key.
	 * @param int    $product_id The product id.
	 * @param int    $quantity The quantity.
	 * @param int    $variation_id The variation id.
	 * @param array  $variation The variation attributes.
	 * @param array  $cart_item_data The cart item.
	 *
	 * @throws Exception Plugins can throw an exception to prevent adding to cart.
	 * @author YITH
	 * @since 3.0.0
	 */
	public function add_bogo_product( $cart_item_key, $product_id, $quantity, $variation_id, $variation, $cart_item_data ) {
		remove_action( 'woocommerce_add_to_cart', array( $this, 'add_bogo_product' ), 210 );
		remove_action( 'woocommerce_add_to_cart', array( $this, 'print_popup_for_special_offer' ), 200 );
		$bogo_rules        = ywdpd_get_price_rules_by_type( 'bogo' );
		$cart_has_discount = WC()->cart->has_discount();

		if ( isset( WC()->cart->cart_contents[ $cart_item_key ]['data'] ) ) {
			$product    = WC()->cart->cart_contents[ $cart_item_key ]['data'];
			$is_on_sale = $product->is_on_sale();
			$qty_added  = WC()->cart->cart_contents[ $cart_item_key ]['quantity'];
			/**
			 * The current rule.
			 *
			 * @var YWDPD_BOGO $bogo
			 */
			foreach ( $bogo_rules as $bogo ) {
				$is_enabled_for_on_sale_products = ! ( $is_on_sale && $bogo->is_disabled_on_sale() );
				$is_enabled_with_other_coupon    = ! ( $cart_has_discount && $bogo->is_disabled_with_other_coupon() );
				if ( $is_enabled_for_on_sale_products && $is_enabled_with_other_coupon && $bogo->is_valid() && $bogo->is_valid_to_apply( $product ) ) {
					$key = YWDPD_Counter::is_bogo_product_in_list( $cart_item_key );

					if ( $key ) { // BOGO Already added.

						if ( isset( WC()->cart->cart_contents[ $key ] ) ) {
							$qty = WC()->cart->cart_contents[ $key ]['quantity'];
							if ( $qty < $qty_added ) {
								WC()->cart->set_quantity( $key, $qty_added, true );
							}
						} else {
							YWDPD_Counter::remove_bogo_product_in_list( $cart_item_key );
						}
					} else {
						$cart_item_bogo = array(
							'has_bogo_applied'  => true,
							'original_cart_key' => $cart_item_key,
							'rule_id'           => $bogo->get_id(),
						);

						$cart_item_bogo = array_merge( $cart_item_data, $cart_item_bogo );
						$new_key        = WC()->cart->add_to_cart( $product_id, $quantity, $variation_id, $variation, $cart_item_bogo );
						if ( $new_key ) {
							YWDPD_Counter::save_bogo_product_in_list( $cart_item_key, $new_key );
						}
					}
				}
			}
		}
		add_action( 'woocommerce_add_to_cart', array( $this, 'add_bogo_product' ), 210, 6 );
		add_action( 'woocommerce_add_to_cart', array( $this, 'print_popup_for_special_offer' ), 200, 6 );
	}

	/**
	 * Check if in the cart there are special offer that can be applied in popup
	 *
	 * @param array  $cart_item The cart item.
	 * @param string $cart_item_key The cart item key.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0
	 */
	public function get_valid_special_offer_to_apply( $cart_item, $cart_item_key ) {

		$special_offers = array();

		if ( ! is_null( WC()->cart ) && ! WC()->cart->is_empty() ) {
			$special_offers_rules = ywdpd_get_price_rules_by_type( 'special_offer' );
			$product              = $cart_item['data'];
			foreach ( $special_offers_rules as $rule ) {
				/**
				 * The special offer rule
				 *
				 * @var YWDPD_Special_Offer $rule
				 */
				$can_in_popup = $rule->can_show_in_popup();
				if ( $can_in_popup ) {
					$is_valid = $rule->is_valid_to_apply( $product );

					if ( $is_valid ) {

						$total_to_add = $rule->get_total_target( $cart_item_key );

						$is_active_apply_to_another_product = $rule->is_enabled_apply_adjustment_to();

						if ( $is_active_apply_to_another_product ) {

							$total_added = $this->get_total_product_with_special_offer( $rule );

							if ( $total_to_add - $total_added > 0 ) {
								if ( ! in_array( $rule->get_id(), $special_offers, true ) ) {
									$rule_text                         = $rule->get_text_in_modal_special_offer();
									$so_rule                           = $rule->get_so_rule();
									$discount                          = array(
										'type'   => $so_rule['type_discount'],
										'amount' => $so_rule['discount_amount'],
									);
									$special_offer                     = array(
										'text'          => $rule_text,
										'items_in_cart' => $total_added,
										'total_to_add'  => $total_to_add,
										'allowed_item'  => $total_to_add - $total_added,
										'items'         => YWDPD_Utils::get_involved_adjustment_items( $rule ),
										'type'          => 'special_offer',
										'discount'      => $discount,
									);
									$special_offers[ $rule->get_id() ] = $special_offer;
								}
							}
						}
					}
				}
			}
		}

		return $special_offers;
	}

	/**
	 * Get the amount of product with this special offer in cart.
	 *
	 * @param YWDPD_Special_Offer $special_offer The special offer rule.
	 *
	 * @return int
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_total_product_with_special_offer( $special_offer ) {
		return YWDPD_Counter::get_product_with_dynamic_rule_counter( 'special_offer', $special_offer->get_id() );
	}

	/**
	 * Compute the total target for this product.
	 *
	 * @param array $cloned_cart The cloned cart.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function set_total_target( $cloned_cart ) {
		if ( ! is_null( WC()->cart ) && ! WC()->cart->is_empty() && count( $cloned_cart ) > 0 ) {
			$special_offers_rules = ywdpd_get_price_rules_by_type( 'special_offer' );
			foreach ( $special_offers_rules as $special_offer ) {
				foreach ( $cloned_cart as $cart_item_key => $cart_item ) {
					$product = $cart_item['data'];
					if ( $special_offer->is_valid_to_apply( $product ) ) {
						$rule         = $special_offer->get_so_rule();
						$purchase     = isset( $rule['purchase'] ) ? intval( $rule['purchase'] ) : 1;
						$receive      = isset( $rule['receive'] ) ? intval( $rule['receive'] ) : 0;
						$rcq          = $special_offer->num_valid_product_to_apply_in_cart( $cart_item, true ); // remaining clean quantity.
						$rmq          = $special_offer->num_valid_product_to_apply_in_cart( $cart_item ); // remaining mixed quantity.
						$repetitions  = $special_offer->get_total_repetitions( $rcq, $rmq, $purchase );
						$tt           = 0; // Total targets.
						$tot_apply_to = $rmq + $rcq;

						if ( $rcq || $rmq ) {

							for ( $x = 1; $x <= $repetitions; $x ++ ) {
								if ( $tot_apply_to - $purchase >= 0 ) {
									$tot_apply_to -= $purchase;
									$tt           += $receive;
								}
							}
						}

						WC()->cart->cart_contents[ $cart_item_key ][ $special_offer->get_id() ]['total_target'] = $tt;

						if ( $special_offer->is_enabled_apply_adjustment_to() ) {
							foreach ( $cloned_cart as $cart_item_adjust_key => $cart_adjust_item ) {
								if ( $cart_item_adjust_key !== $cart_item_key && $special_offer->is_valid_to_adjust( $cart_adjust_item['data'] ) ) {
									WC()->cart->cart_contents[ $cart_item_adjust_key ][ $special_offer->get_id() ]['total_target'] = $tt;
								}
							}
						}
					}
				}
			}
		}
	}

	/**
	 * Hide the quantity field for the free product added with gift or with BOGO
	 *
	 * @auhtor YITH
	 *
	 * @param string $product_quantity The quantity field.
	 * @param string $cart_item_key The cart item key.
	 * @param array  $cart_item The item.
	 *
	 * @return string
	 * @since 3.0.0
	 */
	public function hide_qty_field( $product_quantity, $cart_item_key, $cart_item ) {
		if ( isset( $cart_item['ywdpd_is_gift_product'] ) || isset( $cart_item['has_bogo_applied'] ) ) {
			$product_quantity = sprintf( '%1$s <input type="hidden" name="cart[%2$s][qty]" value="%1$s" />', $cart_item['quantity'], $cart_item_key );
		}

		return $product_quantity;
	}

	/**
	 * Hide the remove url if the product is a gift or bogo offer.
	 *
	 * @param string $link The remove link.
	 * @param string $cart_item_key The cart item key.
	 *
	 * @return  string
	 * @since 3.0.0
	 * @author YITH
	 */
	public function hide_remove_link( $link, $cart_item_key ) {
		$cart_item = isset( WC()->cart->cart_contents[ $cart_item_key ] ) ? WC()->cart->cart_contents[ $cart_item_key ] : array();
		if ( isset( $cart_item['ywdpd_is_gift_product'] ) || isset( $cart_item['has_bogo_applied'] ) ) {
			$link = '';
		}

		return $link;
	}

	/**
	 * Update the bogo quantity
	 *
	 * @param string $cart_item_key The cart item key.
	 * @param int    $quantity The new quantity.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function update_cart_item_with_bogo_quantity( $cart_item_key, $quantity ) {

		remove_action( 'woocommerce_after_cart_item_quantity_update', array( $this, 'update_cart_item_with_bogo_quantity' ), 10 );

		$cart_bogo_item_key = YWDPD_Counter::is_bogo_product_in_list( $cart_item_key );

		if ( $cart_bogo_item_key ) {
			WC()->cart->set_quantity( $cart_bogo_item_key, $quantity, true );
		}
		add_action( 'woocommerce_after_cart_item_quantity_update', array( $this, 'update_cart_item_with_bogo_quantity' ), 10, 2 );
	}

	/**
	 * Remove the bogo item if the main item has been removed.
	 *
	 * @param string $cart_item_key The cart item key.
	 *
	 * @since 3.0.0
	 * @author YITH
	 */
	public function remove_cart_item_with_bogo( $cart_item_key ) {
		remove_action( 'woocommerce_cart_item_removed', array( $this, 'remove_cart_item_with_bogo' ), 10 );
		$cart_bogo_item_key = YWDPD_Counter::is_bogo_product_in_list( $cart_item_key );
		if ( $cart_bogo_item_key ) {
			WC()->cart->remove_cart_item( $cart_bogo_item_key );
			YWDPD_Counter::remove_bogo_product_in_list( $cart_item_key );
		}
		add_action( 'woocommerce_cart_item_removed', array( $this, 'remove_cart_item_with_bogo' ), 10, 1 );
	}

	/**
	 * Check if the items with the bogo rule are valid or not.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function remove_invalid_bogo() {

		if ( ! is_null( WC()->cart ) ) {

			foreach ( WC()->cart->cart_contents as $cart_item_key => $cart_item ) {
				if ( isset( $cart_item['has_bogo_applied'] ) && isset( $this->applied_rules_by_item_key[ $cart_item_key ] ) ) {
					$rule_ids = $this->applied_rules_by_item_key[ $cart_item_key ];
					$bogo_id  = $cart_item['rule_id'];
					if ( ! in_array( intval( $bogo_id ), array_map( 'intval', $rule_ids ), true ) ) {
						$this->remove_cart_item_with_bogo( $cart_item['original_cart_key'] );
					}
				}
			}
		}
	}

	/**
	 * Exclude bundle product from rules
	 *
	 * @param bool  $included is included.
	 * @param array $cart_item The cart item.
	 *
	 * @return bool
	 * @since 3.3.0
	 * @author YITH
	 */
	public function exclude_item_from_cart( $included, $cart_item ) {

		if ( $included ) {
			$included = ! ( isset( $cart_item['bundled_by'] ) || isset( $cart_item['cartstamp'] ) );
		}

		return $included;
	}
}
