<?php // phpcs:ignore WordPress.Files.FileName
/**
 * The class that manage the cart rules
 *
 * @package  YITH\DynamicPricing\And\Discounts\Classes
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * The class that manage the cart rules
 */
class YWDPD_Cart_Rules_Manager {

	/**
	 * The unique access of the class.
	 *
	 * @var YWDPD_Cart_Rules_Manager
	 */
	protected static $instance;

	/**
	 * The array that contain all cart rules object.
	 *
	 * @var array
	 */
	protected $cart_rules;

	/**
	 * YWDPD_Cart_Rules_Manager constructor.
	 */
	private function __construct() {
		add_action( 'woocommerce_cart_updated', array( $this, 'apply_coupon_cart_discount' ), 99 );
		add_action( 'woocommerce_cart_loaded_from_session', array( $this, 'apply_coupon_cart_discount' ), 101 );
		add_filter( 'woocommerce_cart_totals_coupon_label', array( $this, 'dynamic_label_coupon' ), 10, 2 );
		add_filter( 'woocommerce_cart_totals_coupon_html', array( $this, 'coupon_cart_html' ), 10, 2 );
		add_filter( 'woocommerce_coupon_error', array( $this, 'remove_coupon_cart_message' ), 10, 3 );
		add_filter( 'woocommerce_coupon_message', array( $this, 'remove_coupon_cart_message' ), 10, 3 );
		add_filter( 'ywdpd_get_cart_item_quantities', array( $this, 'remove_items_from_quantities' ), 10, 1 );
		add_filter( 'ywdpd_product_can_be_counted_in_condition', array( $this, 'exclude_special_items' ), 20, 2 );

	}

	/**
	 * Prevent the clone of object
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	private function __clone() {
	}

	/**
	 * Prevent unserialize the object.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function __wakeup() {
	}

	/**
	 * Return the unique instance of the class.
	 *
	 * @return YWDPD_Cart_Rules_Manager
	 * @since 3.0.0
	 * @author YITH
	 */
	public static function get_instance() {
		if ( is_null( self::$instance ) ) {

			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Load the cart rules
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function load_cart_rules() {
		$this->cart_rules = $this->get_valid_rules( ywdpd_get_cart_rules() );
	}

	/**
	 * Get the valid cart rules
	 *
	 * @param array $rules The enabled cart rules.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_valid_rules( $rules ) {
		$valid_rules = array();
		if ( count( $rules ) > 0 ) {
			foreach ( $rules as $cart_rule ) {
				/**
				 * The single rule
				 *
				 * @var YWDPD_Cart_Rule $cart_rule
				 */

				$allow_with_other_coupon = ! ( $this->cart_has_coupons() && ! $cart_rule->can_be_used_with_other_coupons() );

				if ( $cart_rule->is_valid() && $allow_with_other_coupon ) {
					$valid_rules[] = $cart_rule;
				}
			}
		}

		return $valid_rules;
	}

	/**
	 * Check if in the cart there are other coupon applied
	 *
	 * @return bool
	 * @since 3.0.0
	 * @author YITH
	 */
	public function cart_has_coupons() {

		$coupons_applied = 0;

		if ( ! is_null( WC()->cart ) ) {
			$coupons_applied = count( WC()->cart->applied_coupons );

			if ( WC()->cart->has_discount( $this->get_coupon_code() ) ) {
				return false;
			}
		}

		return $coupons_applied > 0;
	}

	/**
	 * Check if the coupon should allow the free shipping
	 *
	 * @param array $valid_rules Valid Cart rules.
	 *
	 * @return bool
	 * @author YITH
	 * @since 3.0.0
	 */
	public function allow_free_shipping( $valid_rules ) {

		$allow_shipping = false;
		foreach ( $valid_rules as $cart_rule ) {

			if ( $cart_rule->allow_free_shipping() ) {
				$allow_shipping = true;
				break;
			}
		}

		return $allow_shipping;
	}

	/**
	 * Apply the coupon in the cart
	 *
	 * @throws Exception The exception.
	 * @since 3.0.0
	 * @author YITH
	 */
	public function apply_coupon_cart_discount() {

		$this->load_cart_rules();
		$coupon_code = $this->get_coupon_code();

		if ( count( $this->cart_rules ) > 0 ) {

			$wc_discount = new WC_Discounts( WC()->cart );
			$coupon      = new WC_Coupon( $coupon_code );
			$valid       = $wc_discount->is_coupon_valid( $coupon );
			$valid       = is_wp_error( $valid ) ? false : $valid;
			$coupon_data = array(
				'discount_type'        => 'fixed_cart',
				'amount'               => '',
				'individual_use'       => false,
				'usage_limit'          => 0,
				'excluded_product_ids' => array(),
				'free_shipping'        => $this->allow_free_shipping( $this->cart_rules ),
			);
			if ( 1 === count( $this->cart_rules ) ) {

				$valid_rule = reset( $this->cart_rules );
				/**
				 * The unique valid rule
				 *
				 * @var YWDPD_Cart_Rule $valid_rule
				 */

				if ( 'percentage' === $valid_rule->get_discount_type() ) {
					$coupon_data['amount']               = $valid_rule->get_discount_amount();
					$coupon_data['excluded_product_ids'] = $this->get_excluded_product_ids( $valid_rule );
					$coupon_data['discount_type']        = 'percent';
				} else {
					$coupon_data['amount'] = $this->get_total_discount( $this->cart_rules );
				}
			} else {
				$coupon_data ['amount'] = $this->get_total_discount( $this->cart_rules );
			}

			if ( $valid ) {
				foreach ( $coupon_data as $data_key => $data_value ) {
					if ( is_callable( array( $coupon, 'set_' . $data_key ) ) ) {
						$method = 'set_' . $data_key;
						$coupon->$method( $data_value );
					}
				}
			} else {
				$coupon->add_meta_data( 'ywdpd_coupon', 1 );
				$coupon->add_meta_data( 'ywdpd_coupon_version', '3.0.0' );
				$coupon->read_manual_coupon( $coupon_code, $coupon_data );
			}

			if ( ! $valid || $coupon->get_changes() ) {
				$coupon->save();
			}
			if ( ! WC()->cart->has_discount( $coupon_code ) ) {
				WC()->cart->add_discount( $coupon_code );
			}
		} else {
			$this->remove_all_dynamic_coupons();
		}
	}

	/**
	 * Remove the dynamic coupon
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function remove_all_dynamic_coupons() {
		$applied_coupons = WC()->cart->get_applied_coupons();

		foreach ( $applied_coupons as $applied_coupon ) {
			$coupon   = new WC_Coupon( $applied_coupon );
			$is_ywdpd = $coupon->get_meta( 'ywdpd_coupon', true );

			if ( $is_ywdpd ) {
				WC()->cart->remove_coupon( $applied_coupon );
			}
		}
	}

	/**
	 * Get the total of discount
	 *
	 * @param YWDPD_Cart_Rule[] $rules All cart rules.
	 *
	 * @return float
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_total_discount( $rules ) {
		$discount = 0;

		foreach ( $rules as $cart_rule ) {

			$single_discount = 0;
			$single_subtotal = $cart_rule->get_cart_subtotal();
			$discount_type   = $cart_rule->get_discount_type();
			$discount_amount = $cart_rule->get_discount_amount();
			if ( 'percentage' === $discount_type ) {
				$single_discount = ( $single_subtotal * $discount_amount ) / 100;
			} elseif ( 'price' === $discount_type ) {
				$single_discount = $discount_amount;
			} elseif ( 'fixed-price' === $discount_type ) {
				$discount_amount = apply_filters( 'ywdpd_maybe_should_be_converted', $discount_amount );
				$single_discount = ( $single_subtotal - $discount_amount ) > 0 ? ( $single_subtotal - $discount_amount ) : 0;
			}

			$discount += $single_discount;
		}

		return $discount;
	}

	/**
	 * Get all excluded product by cart rule
	 *
	 * @param YWDPD_Cart_Rule $rule The rule.
	 *
	 * @return array
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_excluded_product_ids( $rule ) {
		$product_ids = array();

		foreach ( WC()->cart->get_cart_contents() as $cart_item_key => $cart_item ) {
			$product = $cart_item['data'];
			if ( $rule->is_product_excluded_from_conditions( $product ) ) {

				$product_id = $product->get_id();
				if ( ! in_array( $product_id, $product_ids ) ) { // phpcs:ignore
					$product_ids[] = $product_id;
				}
			}
		}

		return $product_ids;
	}

	/**
	 * Get the coupon code
	 *
	 * @return string
	 * @since 3.0.0
	 * @author YITH
	 */
	public function get_coupon_code() {
		$label_coupon = 'ywdpd_discount';
		if ( is_user_logged_in() ) {
			$current_coupon_code = apply_filters( 'ywdpd_coupon_code', $label_coupon . '_' . get_current_user_id(), $label_coupon );
			$session             = WC()->session->get( 'ywdpd_coupon_code', '' );

			if ( ! empty( $session ) ) {

				$coupon = new WC_Coupon( $session );
				$coupon->delete( true );
				WC()->session->set( 'ywdpd_coupon_code', '' );
				WC()->session->save_data();
				WC()->cart->remove_coupon( $session );
			}
		} else {
			$session = WC()->session->get( 'ywdpd_coupon_code', '' );
			if ( empty( $session ) ) {
				$current_coupon_code = apply_filters( 'ywdpd_coupon_code', uniqid( $label_coupon . '_' ), $label_coupon );
				WC()->session->set( 'ywdpd_coupon_code', $current_coupon_code );
				WC()->session->save_data();
			} else {
				$current_coupon_code = $session;
			}
		}

		return $current_coupon_code;
	}

	/**
	 * Change the dynamic coupon label
	 *
	 * @param string    $string Coupon code.
	 * @param WC_Coupon $coupon The coupon.
	 *
	 * @return string
	 * @since 3.0.0
	 *
	 * @author YITH
	 */
	public function dynamic_label_coupon( $string, $coupon ) {

		if ( is_null( $coupon ) || ! is_object( $coupon ) ) {
			return $string;
		}

		$is_ywdpd = $coupon->get_meta( 'ywdpd_coupon', true );

		$coupon_label = $this->get_coupon_code_details( true );

		$coupon_label = apply_filters( 'ywdpd_dynamic_label_coupon', $coupon_label, $coupon );

		return $is_ywdpd ? $coupon_label : $string;
	}

	/**
	 * Delete the remove link in the coupon
	 *
	 * @param string    $value The old html value.
	 * @param WC_Coupon $coupon The coupon.
	 *
	 * @return string
	 * @since 3.0.0
	 *
	 * @author YITh
	 */
	public function coupon_cart_html( $value, $coupon ) {
		$is_ywdpd = $coupon->get_meta( 'ywdpd_coupon', true );
		if ( $is_ywdpd ) {

			$amount = WC()->cart->get_coupon_discount_amount( $coupon->get_code(), WC()->cart->display_cart_ex_tax );
			$value  = '-' . wc_price( $amount );

		}

		return $value;
	}

	/**
	 * Remove coupon cart message.
	 *
	 * @param string    $error Error Message.
	 * @param string    $msg_code Code message.
	 * @param WC_Coupon $coupon Coupon.
	 *
	 * @return bool
	 */
	public function remove_coupon_cart_message( $error, $msg_code, $coupon ) {
		if ( $coupon instanceof WC_Coupon ) {
			$is_ywdpd = $coupon->get_meta( 'ywdpd_coupon', true );
			if ( $is_ywdpd ) {
				return false;
			}
		}

		return $error;
	}

	/**
	 * Remove from the quantity, the product with a gift or bogo rule
	 *
	 * @auhtor YITh
	 *
	 * @param int $num_items The items.
	 *
	 * @return int
	 * @since 3.0.0
	 */
	public function remove_items_from_quantities( $num_items ) {
		$items_to_remove = 0;

		foreach ( WC()->cart->get_cart_contents() as $cart_item_key => $cart_item ) {

			if ( isset( $cart_item['ywdpd_is_gift_product'] ) || isset( $cart_item['has_bogo_applied'] ) ) {
				$items_to_remove ++;
			}
		}

		if ( is_array( $num_items ) ) {
			$num_items = array_sum( $num_items );
		}

		return $num_items - $items_to_remove;
	}

	/**
	 * Exclude special items from cart rule condition
	 *
	 * @param bool  $is_valid is valid.
	 * @param array $cart_item The cart item.
	 *
	 * @author YITH
	 * @since 3.0.0
	 */
	public function exclude_special_items( $is_valid, $cart_item ) {

		if ( $is_valid ) {
			$is_valid = ! ( isset( $cart_item['ywdpd_is_gift_product'] ) || isset( $cart_item['has_bogo_applied'] ) );
		}

		return $is_valid;
	}

	/**
	 * Build the right coupon code
	 *
	 * @param bool $formatted Return the coupon code formatted or simple.
	 *
	 * @return string
	 * @author YITH
	 * @since 3.0.0
	 */
	public function get_coupon_code_details( $formatted = false ) {
		$label = '';
		if ( 'default_name' === YITH_WC_Dynamic_Options::get_coupon_label_mode() ) {
			$label = YITH_WC_Dynamic_Options::get_coupon_label();
			$label = $formatted ? $label . ':' : str_replace( ' ', '-', strtolower( $label ) );
		} else {

			if ( count( $this->cart_rules ) > 0 ) {

				foreach ( $this->cart_rules as $cart_rule ) {
					$name = $formatted ? '<li>' . $cart_rule->get_name() . '</li>' : str_replace( ' ', '-', strtolower( $cart_rule->get_name() ) ) . ', ';

					$label .= $name;
				}
				if ( $formatted ) {
					$label = '<ul class="ywdpd_list_cart_rules_applied">' . $label . '</ul>';
				}
			}
		}

		return $label;
	}
}
