<?php
/**
 * This class manage the admin area
 *
 * @package YITH\DynamicPricingAndDiscounts\Classes
 */

if ( ! defined( 'ABSPATH' ) || ! defined( 'YITH_YWDPD_VERSION' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Implements admin features of YITH WooCommerce Dynamic Pricing and Discounts
 *
 * @class   YITH_WC_Dynamic_Pricing_Admin
 * @package YITH WooCommerce Dynamic Pricing and Discounts
 * @since   1.0.0
 * @author  YITH
 */
if ( ! class_exists( 'YITH_WC_Dynamic_Pricing_Admin' ) ) {

	/**
	 * Class YITH_WC_Dynamic_Pricing_Admin
	 */
	class YITH_WC_Dynamic_Pricing_Admin {

		/**
		 * Single instance of the class
		 *
		 * @var YITH_WC_Dynamic_Pricing_Admin
		 */
		protected static $instance;

		/**
		 * This is the instance of the YIT WooCommerce Panel class.
		 *
		 * @var YIT_Plugin_Panel_WooCommerce
		 */
		protected $panel;

		/**
		 * Panel page
		 *
		 * @var string
		 */
		protected $panel_page = 'yith_woocommerce_dynamic_pricing_and_discounts';

		/**
		 * Post type name.
		 *
		 * @var string Doc Url
		 */
		public $post_type_name = 'ywdpd_discount';

		/**
		 * Returns single instance of the class
		 *
		 * @return YITH_WC_Dynamic_Pricing_Admin
		 * @since 1.0.0
		 */
		public static function get_instance() {
			if ( is_null( self::$instance ) ) {
				self::$instance = new self();
			}

			return self::$instance;
		}

		/**
		 * Constructor
		 *
		 * Initialize plugin and registers actions and filters to be used
		 *
		 * @since  1.0.0
		 * @author Emanuela Castorina
		 */
		public function __construct() {

			$this->create_menu_items();

			// register plugin to licence/update system.
			add_action( 'wp_loaded', array( $this, 'register_plugin_for_activation' ), 99 );
			add_action( 'admin_init', array( $this, 'register_plugin_for_updates' ) );

			// Add action links.
			add_filter( 'plugin_action_links_' . plugin_basename( YITH_YWDPD_DIR . '/' . basename( YITH_YWDPD_FILE ) ), array( $this, 'action_links' ) );
			add_filter( 'yith_show_plugin_row_meta', array( $this, 'plugin_row_meta' ), 10, 5 );

			add_action( 'admin_init', array( $this, 'add_metabox' ) );
			add_action( 'admin_action_duplicate_discount', array( $this, 'duplicate_discount' ), 30 );
			add_action( 'admin_action_create_dynamic_rule', array( $this, 'create_dynamic_rule' ), 30 );
			add_action( 'admin_action_create_simple_dynamic_rule', array( $this, 'create_simple_dynamic_rule' ), 30 );
			add_action( 'admin_init', array( $this, 'delete_item_from_exclusion_list' ), 30 );
			add_filter( 'enter_title_here', array( $this, 'change_new_post_placeholder' ), 10, 2 );
			add_filter( 'yit_fw_metaboxes_type_args', array( $this, 'add_custom_type_metaboxes' ) );
			add_action( 'admin_menu', array( $this, 'remove_publish_box' ) );

			// delete transient on rule save or insert.
			add_action( 'save_post', array( $this, 'delete_transient' ), 10, 2 );
			add_action( 'delete_post', array( $this, 'delete_transient' ), 10, 2 );
			add_action( 'wp_insert_post', array( $this, 'delete_transient' ), 10, 2 );
			add_filter( 'redirect_post_location', array( $this, 'add_query_args_edit_cart_rule' ), 20, 2 );
			add_filter( 'post_updated_messages', array( $this, 'post_updated_messages' ) );
			add_filter( 'bulk_post_updated_messages', array( $this, 'bulk_post_updated_messages' ), 10, 2 );

			add_action( 'admin_footer', array( $this, 'show_admin_popup' ), 25 );

			add_action( 'yith_ywdpd_exclusions_table', array( $this, 'show_the_exclusion_table' ) );

			add_action( 'admin_init', array( $this, 'add_items_in_exclusion_list' ), 20 );

			add_action( 'woocommerce_after_order_itemmeta', array( $this, 'show_custom_meta_item_in_order' ), 20, 3 );
			add_filter( 'yith_plugin_fw_yith-plugins_page_yith_woocommerce_dynamic_pricing_and_discounts_bulk_delete_confirmation_enabled', '__return_true' );
			add_filter( 'woocommerce_order_item_get_code', array( $this, 'show_right_coupon_info' ), 20, 2 );
		}

		/**
		 * Return the private property panel page.
		 *
		 * @return string
		 */
		public function get_panel_page() {
			return $this->panel_page;
		}


		/**
		 * Clone the rule
		 */
		public function duplicate_discount() {

			if ( empty( $_REQUEST['post'] ) ) {
				wp_die( esc_html__( 'No rule to duplicate has been supplied!', 'ywdpd' ) );
			}

			$rule_id   = isset( $_REQUEST['post'] ) ? absint( $_REQUEST['post'] ) : '';
			$rule_type = isset( $_REQUEST['ywdpd_discount_type'] ) ? sanitize_text_field( wp_unslash( $_REQUEST['ywdpd_discount_type'] ) ) : 'pricing';

			check_admin_referer( 'ywdpd-duplicate-rule_' . $rule_id );

			global $wpdb;

			$query = $wpdb->prepare( "SELECT * from $wpdb->posts WHERE id=%d", $rule_id ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching

			$results        = $wpdb->get_results( $query, ARRAY_A ); // phpcs:ignore WordPress.DB.DirectDatabaseQuery.NoCaching, WordPress.DB.DirectDatabaseQuery.DirectQuery,WordPress.DB.PreparedSQL.NotPrepared
			$admin_post_url = admin_url( 'post.php' );
			$url_args       = array(
				'action'              => 'edit',
				'ywdpd_discount_type' => $rule_type,
			);
			if ( $results ) {
				foreach ( $results as $result ) {
					if ( 'ywdpd_discount' === $result['post_type'] ) {
						unset( $result['ID'] );
						$result['post_title'] .= ' ' . __( '(Copy)', 'ywdpd' );

						$new_post  = wp_insert_post( $result );
						$post_meta = get_post_custom( $rule_id );
						// set unique key and correct post id.
						$post_meta['_key'][0]      = uniqid();
						$post_meta['id'][0]        = $new_post;
						$post_meta['_priority'][0] = ywdpd_get_last_priority('price') + 1;

						if ( is_array( $post_meta ) ) {
							foreach ( $post_meta as $k => $v ) {
								update_post_meta( $new_post, $k, maybe_unserialize( $v[0] ) );
							}
						}
						$url_args['post'] = $new_post;
						$admin_post_url   = esc_url_raw( add_query_arg( $url_args, $admin_post_url ) );
						wp_safe_redirect( $admin_post_url );
					}
				}
			}

			exit;
		}

		/**
		 * Add metabox into ywdpd_discount editor page
		 *
		 * @since 1.4.0
		 * @author Emanuela Castorina
		 */
		public function add_metabox() {

			if ( ywdpd_check_valid_admin_page( $this->post_type_name ) ) {
				$type   = false;
				$posted = $_REQUEST; // phpcs:disable WordPress.Security.NonceVerification.Recommended

				if ( isset( $posted['ywdpd_discount_type'] ) ) {
					$type = $posted['ywdpd_discount_type'];

				} elseif ( isset( $posted['action'] ) && 'yith_plugin_fw_save_toggle_element_metabox' === $posted['action'] ) {
					if ( isset( $posted['yit_metaboxes']['_cart_discount_rules'] ) ) {
						$type = 'cart';
					}
				}

				if ( ! $type && isset( $_POST['yit_metaboxes']['_discount_type'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification
					$type = sanitize_text_field( wp_unslash( $_POST['yit_metaboxes']['_discount_type'] ) ); // phpcs:ignore WordPress.Security.NonceVerification
				}
				if ( $type ) {

					$args = require_once YITH_YWDPD_DIR . 'plugin-options/metabox/ywdpd_' . $type . '_discount.php';

					if ( ! function_exists( 'YIT_Metabox' ) ) {
						require_once 'plugin-fw/yit-plugin.php';
					}

					$metabox = YIT_Metabox( 'ywdpd_' . $type . '_discount' );

					$metabox->init( $args );
				}
			}
		}

		/**
		 * Metabox to show the action of the current discount
		 *
		 * @access public
		 *
		 * @param WP_Post $post Post.
		 *
		 * @return void
		 * @since  1.4.0
		 */
		public function show_discount_action_metabox( $post ) {
			wc_get_template( 'metabox_discount_action_content.php', array(), '', YITH_YWDPD_TEMPLATE_PATH . 'admin/metaboxes/' );
		}


		/**
		 * Returns the panel page URI
		 *
		 * @param string $tab The current tab.
		 *
		 * @return string
		 *
		 * @since 1.4.0
		 * @author Emanuela Castorina <emanuela.castorina@yithemes.com>
		 */
		public function get_panel_page_uri( $tab = '' ) {
			$panel_uri = add_query_arg( 'page', $this->panel_page, admin_url( 'admin.php' ) );
			if ( $tab ) {
				$panel_uri = add_query_arg( 'tab', $tab, $panel_uri );
			}

			return $panel_uri;
		}

		/**
		 * Remove publish box from single page page of ywdpd_discount
		 *
		 * @access public
		 *
		 * @return void
		 * @since  1.4.0
		 * @author Emanuela Castorina <emanuela.castorina@yithemes.com>
		 */
		public function remove_publish_box() {
			remove_meta_box( 'submitdiv', $this->post_type_name, 'side' );
		}

		/**
		 * Shows custom metabox type
		 *
		 * @param array $args Arguments.
		 *
		 * @return array
		 *
		 * @since 1.4.0
		 * @author Emanuela Castorina <emanuela.castorina@yithemes.com>
		 */
		public function add_custom_type_metaboxes( $args ) {

			if ( ywdpd_check_valid_admin_page( $this->post_type_name ) ) {

				$custom_types = array(
					'customers',
					'quantity_discount',
					'special_offer_discount',
					'cart_discount',
					'cart_discount_type',
					'gift_items_in_cart',
					'schedule_rules',
					'quantity_category_discount',
					'save_rule',
					'advanced-simple-text',
				);

				if ( in_array( $args['type'], $custom_types, true ) ) {
					$args['basename'] = YITH_YWDPD_DIR;
					$args['path']     = 'admin/metaboxes/types/';
				}
			}

			return $args;
		}

		/**
		 * Change the url Move to trash to Delete the Discount definitely
		 *
		 * @param string  $url Url.
		 * @param integer $post_id Post id.
		 * @param string  $type Type of discount.
		 *
		 * @return string
		 * @since  1.4.0
		 * @deprecated 1.6.0
		 */
		public function get_delete_post_link( $url, $post_id, $type ) {

			wc_deprecated_function( 'YITH_WC_Dynamic_Pricing_Admin()->get_delete_post_link', '1.6.0', 'get_delete_post_link' );

			$post_type = get_post_type( $post_id );
			if ( $post_type !== $this->post_type_name ) {
				return $url;
			}

			$action      = 'delete';
			$delete_link = add_query_arg( 'action', $action, admin_url( 'admin.php' ) );
			$delete_link = add_query_arg( 'page', $this->panel_page, $delete_link );
			$delete_link = add_query_arg( 'tab', $type, $delete_link );
			$delete_link = add_query_arg( 'post', $post_id, $delete_link );
			$delete_link = wp_nonce_url( $delete_link, "$action-post_{$post_id}" );

			return $delete_link;

		}


		/**
		 * Modify the capability
		 *
		 * @param string $capability Capability.
		 *
		 * @return string
		 */
		public function change_capability( $capability ) {
			return 'manage_woocommerce';
		}


		/**
		 * Create Menu Items
		 *
		 * Print admin menu items
		 *
		 * @since  1.0
		 * @author Emanuela Castorina
		 */
		private function create_menu_items() {
			// Add a panel under YITH Plugins tab.
			add_action( 'admin_menu', array( $this, 'register_panel' ), 5 );
		}

		/**
		 * Add a panel under YITH Plugins tab
		 *
		 * @return   void
		 * @since    1.0
		 * @author   Andrea Grillo <andrea.grillo@yithemes.com>
		 * @use      /Yit_Plugin_Panel class
		 * @see      plugin-fw/lib/yit-plugin-panel.php
		 */
		public function register_panel() {

			if ( ! empty( $this->panel ) ) {
				return;
			}

			$admin_tabs = array(
				'general' => __( 'Settings', 'ywdpd' ),
			);

			if ( defined( 'YITH_YWDPD_FREE_INIT' ) ) {
				$admin_tabs['premium'] = __( 'Premium Version', 'ywdpd' );
			} else {
				$admin_tabs['pricing']        = __( 'Dynamic rules', 'ywdpd' );
				$admin_tabs['quantity-table'] = __( 'Quantity table', 'ywdpd' );
				$admin_tabs['exclusions']     = __( 'Exclusion list', 'ywdpd' );
			}

			$args = array(
				'create_menu_page' => true,
				'parent_slug'      => '',
				'plugin_slug'      => YITH_YWDPD_SLUG,
				'page_title'       => _x( 'YITH WooCommerce Dynamic Pricing and Discounts Premium', 'Plugin name, do not translate', 'ywdpd' ),
				'menu_title'       => 'Dynamic Pricing and Discounts',
				'capability'       => 'manage_options',
				'parent'           => 'ywdpd',
				'parent_page'      => 'yith_plugin_panel',
				'page'             => $this->panel_page,
				'admin-tabs'       => $admin_tabs,
				'options-path'     => YITH_YWDPD_DIR . '/plugin-options',
				'class'            => yith_set_wrapper_class(),
				'help_tab'         => array(
					'main_video' => array(
						'desc' => _x( 'Check this video to learn how to <b>create dynamic rules</b>', '[HELP TAB] Video title', 'ywdpd' ),
						'url'  => array(
							'en' => '//youtube.com/embed/ayLe79SfXto',
							'it' => '//youtube.com/embed/Glrkj1UaD0g',
							'es' => '//youtube.com/embed/vjoAcaMkrL4',
						),
					),
					'playlists'  => array(
						'en' => '//youtube.com/watch?v=ayLe79SfXto&list=PLDriKG-6905n3DUEI6_S217zVl6yNoNzY&ab_channel=YITH',
						'it' => '//youtube.com/watch?v=Glrkj1UaD0g&list=PL9c19edGMs098ev6sVFZRtE7hLyxSaqaL&ab_channel=YITHITALIA',
						'es' => '//youtube.com/watch?v=vjoAcaMkrL4&list=PL9Ka3j92PYJPp5kYy3UV03PRv0VL4TZ1S&ab_channel=YITHESPA%C3%91A',
					),
					'hc_url'     => '//support.yithemes.com/hc/en-us/categories/4410223944209-YITH-WOOCOMMERCE-DYNAMIC-PRICING-AND-DISCOUNTS',
					'doc_url'    => '//docs.yithemes.com/yith-woocommerce-dynamic-pricing-and-discounts/',
				),
			);

			// enable shop manager to set Dynamic Pricing Options.
			$enable_shop_manager = YITH_WC_Dynamic_Options::is_shop_manager_enabled();
			if ( $enable_shop_manager ) {
				add_filter(
					'option_page_capability_yit_' . $args['parent'] . '_options',
					array(
						$this,
						'change_capability',
					)
				);
				$args['capability'] = 'manage_woocommerce';
			}

			/* === Fixed: not updated theme  === */
			if ( ! class_exists( 'YIT_Plugin_Panel_WooCommerce' ) ) {
				require_once YITH_YWDPD_DIR . '/plugin-fw/lib/yit-plugin-panel-wc.php';
			}

			$this->panel = new YIT_Plugin_Panel_WooCommerce( $args );

		}


		/**
		 * Add the action links to plugin admin page
		 *
		 * @param array $links Links plugin array.
		 *
		 * @return mixed
		 * @use      plugin_action_links_{$plugin_file_name}
		 * @author   Andrea Grillo <andrea.grillo@yithemes.com>
		 * @since    1.0
		 */
		public function action_links( $links ) {
			$links = yith_add_action_links( $links, $this->panel_page, true );

			return $links;
		}

		/**
		 * Plugin_row_meta.
		 *
		 * Add the action links to plugin admin page.
		 *
		 * @param array  $new_row_meta_args The new plugin meta.
		 * @param array  $plugin_meta The plugin meta.
		 * @param string $plugin_file The filename of plugin.
		 * @param array  $plugin_data The plugin data.
		 * @param string $status The plugin status.
		 * @param string $init_file The filename of plugin.
		 *
		 * @return   array
		 * @since    1.0
		 * @author  YITH
		 * @use plugin_row_meta
		 */
		public function plugin_row_meta( $new_row_meta_args, $plugin_meta, $plugin_file, $plugin_data, $status, $init_file = 'YITH_YWDPD_INIT' ) {
			if ( defined( $init_file ) && constant( $init_file ) === $plugin_file ) {
				$new_row_meta_args['slug']       = YITH_YWDPD_SLUG;
				$new_row_meta_args['is_premium'] = true;
			}

			return $new_row_meta_args;
		}

		/**
		 * Register plugins for activation tab
		 *
		 * @return void
		 * @since    2.0.0
		 * @author   Andrea Grillo <andrea.grillo@yithemes.com>
		 */
		public function register_plugin_for_activation() {
			if ( ! class_exists( 'YIT_Plugin_Licence' ) ) {
				require_once YITH_YWDPD_DIR . 'plugin-fw/licence/lib/yit-licence.php';
				require_once YITH_YWDPD_DIR . 'plugin-fw/licence/lib/yit-plugin-licence.php';
			}
			YIT_Plugin_Licence()->register( YITH_YWDPD_INIT, YITH_YWDPD_SECRET_KEY, YITH_YWDPD_SLUG );
		}

		/**
		 * Register plugins for update tab
		 *
		 * @return void
		 * @since    2.0.0
		 * @author   Andrea Grillo <andrea.grillo@yithemes.com>
		 */
		public function register_plugin_for_updates() {
			if ( ! class_exists( 'YIT_Upgrade' ) ) {
				require_once YITH_YWDPD_DIR . 'plugin-fw/lib/yit-upgrade.php';
			}
			YIT_Upgrade()->register( YITH_YWDPD_SLUG, YITH_YWDPD_INIT );
		}

		/**
		 * Delete transient on post save or insert
		 *
		 * @param integer $post_id Post id.
		 * @param WP_Post $post Post.
		 *
		 * @return void
		 * @author Francesco Licandro
		 * @since 1.0.0
		 */
		public function delete_transient( $post_id, $post ) {
			if ( 'ywdpd_discount' === $post->post_type ) {

				ywdpd_delete_plugin_transient();
				$rule_version = get_post_meta( $post_id, '_version', true );
				if ( empty( $rule_version ) && 'delete_post' !== current_action() ) {
					$rule_version = '2.0.0';
					update_post_meta( $post_id, '_version', $rule_version );
				}

				if ( isset( $_REQUEST['save'] ) ) {
					if ( 'draft' === get_post_status( $post_id ) ) {
						wp_update_post(
							array(
								'ID'          => $post_id,
								'post_status' => 'publish',
							)
						);
					}
				}
			}
		}

		/**
		 * Add the right query args in edit url
		 *
		 * @param string $edit_url The edit url.
		 * @param int    $post_id The post id.
		 *
		 * @return string
		 * @since 2.0.0
		 * @author YITH
		 */
		public function add_query_args_edit_cart_rule( $edit_url, $post_id ) {

			if ( 'ywdpd_discount' === get_post_type( $post_id ) ) {

				$discount_type = get_post_meta( $post_id, '_discount_type', true );

				$edit_url = add_query_arg( array( 'ywdpd_discount_type' => $discount_type ), $edit_url );
			}

			return $edit_url;
		}

		/**
		 * Change messages when a post type is updated.
		 *
		 * @param array $messages Array of messages.
		 *
		 * @return array
		 * @since 1.3.0
		 * @author YITH
		 */
		public function post_updated_messages( $messages ) {
			global $post;

			$messages['ywdpd_discount'] = array(
				0  => '', // Unused. Messages start at index 1.
				1  => __( 'Dynamic rule updated.', 'ywdpd' ),
				2  => __( 'Dynamic rule updated.', 'ywdpd' ),
				3  => __( 'Dynamic rule deleted.', 'ywdpd' ),
				4  => __( 'Dynamic rule updated.', 'ywdpd' ),
				5  => __( 'Dynamic rule restored.', 'ywdpd' ),
				6  => __( 'Dynamic rule updated.', 'ywdpd' ),
				7  => __( 'Dynamic rule saved.', 'ywdpd' ),
				8  => __( 'Dynamic rule submitted.', 'ywdpd' ),
				9  => sprintf(
				/* translators: %s: date */
					__( 'Dynamic rule scheduled for: %s.', 'yith-woocommerce-role-based-prices' ),
					'<strong>' . date_i18n( __( 'M j, Y @ G:i', 'ywdpd' ), strtotime( $post->post_date ) ) . '</strong>'
				),
				10 => __( 'Dynamic rule draft updated.', 'ywdpd' ),
				11 => __( 'Dynamic rule updated and sent.', 'ywdpd' ),
			);

			return $messages;
		}

		/**
		 * Specify custom bulk actions messages for different post types.
		 *
		 * @param array $bulk_messages Array of messages.
		 * @param array $bulk_counts Array of how many objects were updated.
		 *
		 * @return array
		 * @since 1.3.0
		 * @author YITH
		 */
		public function bulk_post_updated_messages( $bulk_messages, $bulk_counts ) {
			$bulk_messages['ywdpd_discount'] = array(
				/* translators: %s:  dynamic rule count */
				'updated'   => _n( '%s dynamic rule updated.', '%s dynamic rules updated.', $bulk_counts['updated'], 'ywdpd' ),
				/* translators: %s:  dynamic rule count */
				'locked'    => _n( '%s dynamic rule not updated, somebody is editing it.', '%s dynamic rules not updated, somebody is editing them.', $bulk_counts['locked'], 'ywdpd' ),
				/* translators: %s:  dynamic rule count */
				'deleted'   => _n( '%s dynamic rule permanently deleted.', '%s dynamic rules permanently deleted.', $bulk_counts['deleted'], 'ywdpd' ),
				/* translators: %s: dynamic rule count */
				'trashed'   => _n( '%s dynamic rule moved to the Trash.', '%s dynamic rules moved to the Trash.', $bulk_counts['trashed'], 'ywdpd' ),
				/* translators: %s:  dynamic rule count */
				'untrashed' => _n( '%s dynamic rule restored from the Trash.', '%s dynamic rules restored from the Trash.', $bulk_counts['untrashed'], 'ywdpd' ),
			);

			return $bulk_messages;
		}

		/**
		 * Add the popup in the admin area
		 *
		 * @author YITH
		 * @since 3.0.0
		 */
		public function show_admin_popup() {

			$first_check  = isset( $_GET['post_type'] ) && 'ywdpd_discount' === sanitize_text_field( wp_unslash( $_GET['post_type'] ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$second_check = isset( $_GET['post'] ) && 'ywdpd_discount' === get_post_type( sanitize_text_field( wp_unslash( $_GET['post'] ) ) ); // phpcs:ignore WordPress.Security.NonceVerification.Recommended
			if ( $first_check || $second_check ) {
				require_once YITH_YWDPD_TEMPLATE_PATH . 'admin/dynamic-rules-popup.php';
			}
		}

		/**
		 * Create the default rule
		 *
		 * @author YITH
		 * @since 3.0.0
		 */
		public function create_dynamic_rule() {

			if ( ! empty( $_GET['type'] ) ) {

				$type = sanitize_text_field( wp_unslash( $_GET['type'] ) );
				check_admin_referer( 'ywdpd-create-pop-rule-' . $type );
				$discount_type = 'pricing';
				$post_meta     = array(
					'_active'   => 'no',
					'_priority' => ywdpd_get_last_priority( '' ) + 1,
				);

				$new_post_id = wp_insert_post(
					array(
						'post_type'   => 'ywdpd_discount',
						'post_status' => 'publish',
						'post_title'  => $this->generate_dynamic_rule_title( $type ),
					)
				);

				if ( 'three_x_two' === $type ) {
					$post_meta['_discount_mode'] = 'special_offer';
					$post_meta['_rule_for']      = 'specific_products';
					$post_meta['so-rule']        = array(
						'purchase'        => 3,
						'receive'         => 1,
						'type_discount'   => 'percentage',
						'discount_amount' => 100,
					);
				} elseif ( 'two_x_one' === $type ) {
					$post_meta['_discount_mode'] = 'special_offer';
					$post_meta['_rule_for']      = 'specific_products';
					$post_meta['so-rule']        = array(
						'purchase'        => 2,
						'receive'         => 1,
						'type_discount'   => 'percentage',
						'discount_amount' => 100,
					);
				} elseif ( 'bogo' === $type ) {
					$post_meta['_discount_mode'] = 'bogo';
					$post_meta['_rule_for']      = 'specific_products';
				} elseif ( 'off_2dn_unit' === $type ) {
					$post_meta['_discount_mode'] = 'special_offer';
					$post_meta['_rule_for']      = 'specific_products';
					$post_meta['so-rule']        = array(
						'purchase'        => 2,
						'receive'         => 1,
						'type_discount'   => 'percentage',
						'discount_amount' => 30,
					);
				} elseif ( 'scheduled' === $type ) {
					$post_meta['_discount_mode']          = 'discount_whole';
					$post_meta['_schedule_discount_mode'] = array(
						'schedule_type' => 'schedule_dates',
						'schedule_from' => gmdate( 'Y-m-d H:i' ),
						'schedule_to'   => gmdate( 'Y-m-d H:i', strtotime( '+4 days' ) ),
					);
				} elseif ( 'quantity_table' === $type ) {
					$post_meta['_discount_mode'] = 'bulk';
				} elseif ( 'user' === $type ) {
					$post_meta['_discount_mode'] = 'discount_whole';
					$post_meta['_user_rules']    = 'role_list';
				} elseif ( 'buy_x_get_y' === $type ) {
					$post_meta['_discount_mode']                      = 'special_offer';
					$post_meta['_rule_for']                           = 'specific_products';
					$post_meta['_active_apply_discount_to']           = 'yes';
					$post_meta['_rule_apply_adjustment_discount_for'] = 'specific_products';
					$post_meta['so-rule']                             = array(
						'purchase'        => 1,
						'receive'         => 1,
						'type_discount'   => 'percentage',
						'discount_amount' => 100,
					);
				} elseif ( 'gift_rule' === $type ) {
					$post_meta['_discount_mode'] = 'gift_products';
				} elseif ( 'cart_rule' === $type ) {
					$discount_type                     = 'cart';
					$post_meta['_cart_discount_rules'] = array(
						array(
							'cart_condition_name' => __( 'Simple user condition', 'ywdpd' ),
							'condition_for'       => 'customers',
							'user_discount_to'    => 'all',
						),
					);
				} else {
					$discount_type                     = 'cart';
					$post_meta['_cart_discount_rules'] = array(
						array(
							'cart_condition_name' => __( 'Simple total condition', 'ywdpd' ),
							'condition_for'       => 'cart_subtotal',
							'min_subtotal'        => 100,
						),
					);
					$post_meta['_allow_free_shipping'] = 'yes';
				}

				$post_meta['_discount_type'] = $discount_type;

				foreach ( $post_meta as $meta_key => $meta_value ) {
					update_post_meta( $new_post_id, $meta_key, $meta_value );
				}

				$url = add_query_arg(
					array(
						'post'                => $new_post_id,
						'action'              => 'edit',
						'ywdpd_discount_type' => $discount_type,
					),
					admin_url( 'post.php' )
				);
				wp_safe_redirect( esc_url_raw( $url ) );
				exit;
			}
		}

		/**
		 * Create an empty dynamic rule
		 *
		 * @author YITH
		 * @since 3.0.0
		 */
		public function create_simple_dynamic_rule() {
			if ( ! empty( $_GET['type'] ) ) {
				$type = sanitize_text_field( wp_unslash( $_GET['type'] ) );
				check_admin_referer( 'ywdpd-create-pop-simple-rule-' . $type );
				$discount_type = 'pricing';
				$post_meta     = array(
					'_active'   => 'no',
					'_priority' => ywdpd_get_last_priority( '' ) + 1,
				);

				$new_post_id = wp_insert_post(
					array(
						'post_type'   => 'ywdpd_discount',
						'post_status' => 'publish',
						'post_title'  => $this->generate_dynamic_rule_title( $type ),
					)
				);

				if ( 'simple_qty' === $type ) {
					$post_meta['_discount_mode'] = 'bulk';
				} elseif ( 'simple_special_offer' === $type ) {
					$post_meta['_discount_mode'] = 'special_offer';
				} elseif ( 'simple_gift_products' === $type ) {
					$post_meta['_discount_mode'] = 'gift_products';
				} elseif ( 'simple_whole_shop' === $type ) {
					$post_meta['_discount_mode'] = 'discount_whole';
				} elseif ( 'simple_category_discount' === $type ) {
					$post_meta['_discount_mode'] = 'category_discount';
				} else {
					$discount_type = 'cart';
				}

				$post_meta['_discount_type'] = $discount_type;

				foreach ( $post_meta as $meta_key => $meta_value ) {
					update_post_meta( $new_post_id, $meta_key, $meta_value );
				}

				$url = add_query_arg(
					array(
						'post'                => $new_post_id,
						'action'              => 'edit',
						'ywdpd_discount_type' => $discount_type,
					),
					admin_url( 'post.php' )
				);
				wp_safe_redirect( esc_url_raw( $url ) );
				exit;

			}
		}

		/**
		 * Return the unique rule title
		 *
		 * @param string $type The rule type.
		 *
		 * @return string
		 * @author YITH
		 * @since 3.0.0
		 */
		public function generate_dynamic_rule_title( $type ) {

			$post_title  = str_replace( '_', ' ', $type );
			$check_title = $post_title;
			$counter     = 1;

			while ( post_exists( $check_title, '', '', 'ywdpd_discount' ) ) {
				$check_title = $post_title . ' ' . $counter;
				$counter ++;
			}

			return $check_title;
		}

		/**
		 * Add categories exclusion table.
		 *
		 * @access public
		 * @since  3.0.0
		 * @author YITH
		 */
		public function show_the_exclusion_table() {

			$get           = $_GET; //phpcs:ignore WordPress.Security.NonceVerification.Recommended
			$template_file = YITH_YWDPD_VIEW_PATH . '/panel/custom-tabs/ywdpd-exclusions-table.php';

			if ( isset( $get['page'] ) && 'yith_woocommerce_dynamic_pricing_and_discounts' === $get['page'] && isset( $get['tab'] ) && 'exclusions' === $get['tab'] && file_exists( $template_file ) ) {

				$exclusions_prod = array_filter( explode( ',', get_option( 'yith-ywdpd-exclusions-prod-list' ) ) );
				$exclusions_cat  = array_filter( explode( ',', get_option( 'yith-ywdpd-exclusions-cat-list' ) ) );
				$exclusions_tag  = array_filter( explode( ',', get_option( 'yith-ywdpd-exclusions-tag-list' ) ) );
				$list            = array_merge( $exclusions_prod, $exclusions_cat, $exclusions_tag );

				$is_blank = ! YWDPD_Exclusion_List_Data_Store::has_items();

				wp_enqueue_style( 'ywdpd_exclusion' );
				wp_enqueue_script( 'ywdpd_exclusion' );
				include_once YITH_YWDPD_INC . '/admin/class-ywdpd-exclusions-list-table.php';

				$table = new YWDPD_Exclusions_List_Table();
				$table->prepare_items();

				include_once $template_file;
			}
		}

		/**
		 * Add items in the exclusion table
		 *
		 * @author YITH
		 * @since 3.0.0
		 */
		public function add_items_in_exclusion_list() {
			$posted = $_POST; //phpcs:ignore WordPress.Security.NonceVerification.Missing
			if ( ! isset( $posted['_nonce'], $posted['ywdpd-exclusion-type'] ) || ! wp_verify_nonce( $posted['_nonce'], 'yith_ywdpd_add_exclusions' ) ) {
				return;
			}

			$term_type = $posted['ywdpd-exclusion-type'];
			$term_ids  = $posted[ 'ywdpd_add_' . $term_type ];

			foreach ( $term_ids as $term_id ) {
				YWDPD_Exclusion_List_Data_Store::insert( $term_id, $term_type );
			}
		}

		/**
		 * Delete item from the exclusion list
		 *
		 * @author YITH
		 * @since 3.0.0
		 */
		public function delete_item_from_exclusion_list() {

			if ( isset( $_GET['term_id'] ) && ( isset( $_GET['action'] ) && 'delete_from_exclusion' === $_GET['action'] ) ) {
				$term_id = sanitize_text_field( wp_unslash( $_GET['term_id'] ) );
				check_admin_referer( 'ywdpd-delete-exclusion_' . $term_id );
				YWDPD_Exclusion_List_Data_Store::delete( $term_id );

				$url = add_query_arg(
					array(
						'page' => 'yith_woocommerce_dynamic_pricing_and_discounts',
						'tab'  => 'exclusions',
					),
					admin_url( 'admin.php' )
				);
				wp_safe_redirect( esc_url_raw( $url ) );
				exit;
			}
		}

		/**
		 * Show the dynamic rule in the item
		 *
		 * @param int                   $item_id The item id.
		 * @param WC_Order_Item_Product $item The item.
		 * @param WC_Product            $product The product.
		 *
		 * @since 3.0.0
		 * @author YITH
		 */
		public function show_custom_meta_item_in_order( $item_id, $item, $product = false ) {
			if ( $item instanceof WC_Order_Item_Product ) {
				$dynamic_rules = $item->get_meta( '_ywdpd_discounts' );
				if ( ! empty( $dynamic_rules ) ) {
					$custom_meta = '';

					foreach ( $dynamic_rules['applied_discounts'] as $applied_discount ) {
						if ( isset( $applied_discount['set_id'] ) ) {
							$rule_id = $applied_discount['set_id'];
							$rule    = ywdpd_get_rule( $rule_id );
						} else {
							$rule = $applied_discount['by'];
						}
						if ( $rule instanceof YWDPD_Price_Rule && ! empty( $rule->get_name() ) ) {
							$custom_meta .= '<li>' . $rule->get_name() . '</li>';
						}
					}
					if ( ! empty( $custom_meta ) ) { ?>
                        <div class="view">
                            <table cellspacing="0" class="display_meta">
                                <th><?php esc_html_e( 'Offer applied:', 'ywdpd' ); ?></th>
                                <td>
                                    <ul style="margin:0;"><?php echo wp_kses_post( $custom_meta ); ?>
                                    </ul>
                                </td>
                            </table>
                        </div>
						<?php
					}
				}
			}
		}

		/**
		 * Show the right coupon name in backend
		 *
		 * @param string               $code The coupon code.
		 * @param WC_Order_Item_Coupon $coupon_item The coupon item object.
		 *
		 * @author YITH
		 * @since 3.4.0
		 */
		public function show_right_coupon_info( $code, $coupon_item ) {
			$new_code = $coupon_item->get_meta( '_ywdpd_coupon_info' );

			if ( ! empty( $new_code ) ) {
				$code = $new_code;
			}

			return $code;
		}

		/**
		 * Set the custom placholder for new post
		 *
		 * @param string  $title The default post title.
		 * @param WP_Post $post The post.
		 *
		 * @return string
		 * @since 2.2.1
		 * @author YITH
		 */
		public function change_new_post_placeholder( $title, $post ) {

			if ( 'ywdpd_discount' === get_post_type( $post ) ) {
				$title = _x( 'Add name', 'placeholder for new rule title', 'ywdpd' );
			}

			return $title;
		}

	}
}

/**
 * Unique access to instance of YITH_WC_Dynamic_Pricing_Admin class
 *
 * @return YITH_WC_Dynamic_Pricing_Admin
 */
function YITH_WC_Dynamic_Pricing_Admin() {  //phpcs:ignore

	return YITH_WC_Dynamic_Pricing_Admin::get_instance();
}

YITH_WC_Dynamic_Pricing_Admin();
